#pragma once

#include "error.hpp"

#include <src/expected.hpp>
#include <src/log.hpp>
#include <src/task_context.hpp>

#include <ymod_httpclient/call.h>
#include <http_getter/http_request.h>

namespace collie::services {
namespace retry_condition {

struct AcceptHttp404 {
    bool shouldRetry(std::size_t, const boost::system::error_code&) const noexcept {
        return true;
    }

    bool shouldAccept(const yhttp::response& response) const noexcept {
        return response.status == 404;
    }

    bool shouldRetry(std::size_t, const yhttp::response&) const noexcept {
        return true;
    }
};

struct RetryHttp5xx {
    bool shouldRetry(std::size_t, const boost::system::error_code&) const noexcept {
        return true;
    }

    bool shouldAccept(const yhttp::response& response) const noexcept {
        return response.status == 200;
    }

    bool shouldRetry(std::size_t, const yhttp::response& response) const noexcept {
        return response.status / 100 == 5;
    }
};

struct LimitedRetries {
    std::size_t maxRetries;

    bool shouldRetry(std::size_t attempt, const boost::system::error_code&) const noexcept {
        return attempt < maxRetries;
    }

    bool shouldAccept(const yhttp::response&) const noexcept {
        return false;
    }

    bool shouldRetry(std::size_t attempt, const yhttp::response&) const noexcept {
        return attempt < maxRetries;
    }
};

} // namespace retry_condition

template <class T>
decltype(auto) unwrap(T&& value) {
    return std::forward<T>(value);
}

template <class T>
decltype(auto) unwrap(std::reference_wrapper<T>&& value) {
    return value.get();
}

template <class ... Nested>
struct CompositeRetryCondition {
    std::tuple<Nested ...> nested;

    bool shouldRetry(std::size_t attempt, const boost::system::error_code& ec) {
        return std::apply(
            [&] (const auto& ... nested) { return (unwrap(nested).shouldRetry(attempt, ec) && ...); },
            nested
        );
    }

    bool shouldAccept(const yhttp::response& response) {
        return std::apply(
            [&] (const auto& ... nested) { return (unwrap(nested).shouldAccept(response) || ...); },
            nested
        );
    }

    bool shouldRetry(std::size_t attempt, const yhttp::response& response) {
        return std::apply(
            [&] (const auto& ... nested) { return (unwrap(nested).shouldRetry(attempt, response) && ...); },
            nested
        );
    }
};

template <class ... Nested>
auto makeCompositeRetryCondition(Nested&& ... nested) {
    return CompositeRetryCondition<std::decay_t<Nested> ...> {std::make_tuple(std::forward<Nested>(nested) ...)};
}

template <class RetryCondition>
expected<yhttp::response> performWithRetries(yhttp::simple_call& httpClient,
        const TaskContextPtr& context, const http_getter::Request& request, RetryCondition&& retryCondition) {
    using namespace std::string_literals;

    std::size_t attempt = 0;

    while (true) {
        try {
            boost::system::error_code ec;
            auto response = http_getter::asyncRun(httpClient, context, request, context->yield()[ec]);

            if (ec && !retryCondition.shouldRetry(attempt, ec)) {
                return make_unexpected(error_code(ec));
            } else if (ec) {
                ++attempt;
                continue;
            }

            if (retryCondition.shouldAccept(response)) {
                return response;
            }

            if (!retryCondition.shouldRetry(attempt, response)) {
                LOGDOG_(context->logger(), error,
                    log::message="request failed after all tries",
                    log::request_url=request.request.url,
                    log::response_body=response.body
                );
                return make_unexpected(error_code(Error::httpError));
            }

            ++attempt;
        } catch (const ymod_httpclient::bad_url_error& e) {
            return make_unexpected(error_code(Error::invalidUrl, e.what()));
        }
    }
}

} // namespace collie::services
