#pragma once

#include <src/error_code.hpp>

namespace collie::services::settings {

enum class Error {
    ok,
    badResponse,
    nonRetryableStatus,
    retriesExceeded,
    tvmServiceTicketError
};

} // namespace collie::services::settings

namespace boost::system {

template <>
struct is_error_code_enum<collie::services::settings::Error> : std::true_type {};

} // namespace boost::system

namespace collie::services::settings {

class ErrorCategory final : public boost::system::error_category {
public:
    const char* name() const noexcept override {
        return "collie::services::settings::ErrorCategory";
    }

    std::string message(int value) const override {
        switch (static_cast<Error>(value)) {
            case Error::ok:
                return "ok";
            case Error::badResponse:
                return "bad settings response";
            case Error::nonRetryableStatus:
                return "HTTP response status is not retryable";
            case Error::retriesExceeded:
                return "retries possibilities exceeded";
            case Error::tvmServiceTicketError:
                return "can't get tvm service ticket";
        }
        return "unknown error code: " + std::to_string(value);
    }

    static const ErrorCategory& instance() {
        static const ErrorCategory errorCategory;
        return errorCategory;
    }
};

inline error_code::base_type make_error_code(Error ec) {
    return error_code::base_type(static_cast<int>(ec), ErrorCategory::instance());
}

} // namespace collie::services::settings