#include "settings_client_impl.hpp"
#include "error.hpp"

#include <butil/http/headers.h>
#include <ymod_httpclient/url_encode.h>
#include <yamail/data/deserialization/json_reader.h>

#include <boost/lexical_cast.hpp>

BOOST_FUSION_DEFINE_STRUCT((collie)(services)(settings), CollectAddresses,
    (std::optional<std::string>, collect_addresses)
)

BOOST_FUSION_DEFINE_STRUCT((collie)(services)(settings), SingleSettings,
    (std::optional<collie::services::settings::CollectAddresses>, single_settings)
)

BOOST_FUSION_DEFINE_STRUCT((collie)(services)(settings), SettingsResponse,
    (std::optional<collie::services::settings::SingleSettings>, settings)
)

namespace collie::services::settings {

SettingsClientImpl::SettingsClientImpl(
        const GetClusterClient& getHttpClient,
        const GetTvm2Module& getTvm2Module
    )
    : getHttpClient(getHttpClient)
    , getTvm2Module(getTvm2Module)
    {}

expected<std::string> SettingsClientImpl::getTvmServiceTicket(const TaskContextPtr& context) const {
    std::string ticket;
    const auto ec = getTvm2Module()->get_service_ticket("settings", ticket);
    if (ec) {
        LOGDOG_(context->logger(), error,
            log::message = "failed to get TVM2 service ticket for settings request",
            log::error_code = ec
        );
        return make_unexpected(error_code(Error::tvmServiceTicketError, ec.message()));
    }
    return ticket;
}

expected<bool> SettingsClientImpl::getCollectAddressesField(const TaskContextPtr& context, const Uid& uid) const {
    auto perform {[&](auto&& ticket) {
        using namespace http_getter;
        const auto request = get("/get_profile")
                            .headers(requestId=context->requestId(), serviceTicket=ticket)
                            .getArgs("uid"_arg=uid, "settings_list"_arg="collect_addresses")
                            .make();
        return performWithRetries(context, request, uid);
    }};

    auto parseResult {[&](auto&& response) {
        return parseGetCollectAddressesFieldResult(response);
    }};

    return getTvmServiceTicket(context)
        .bind(std::move(perform))
        .bind(std::move(parseResult));
}

expected<SettingsClientImpl::Response> SettingsClientImpl::performWithRetries(
        const TaskContextPtr& context,
        const http_getter::Request& request,
        const Uid& uid) const {

    using namespace std::string_literals;

    const auto httpClient = getHttpClient();
    const auto logger = logdog::bind(context->logger(), log::uid = uid);

    boost::system::error_code ec;
    const auto response = http_getter::asyncRun(*httpClient, context, request, context->yield()[ec]);
    if (ec) {
        return make_unexpected(error_code(ec));
    }
    if (response.status == 200) {
        return response;
    } 
    if (response.status / 100 != 5) {
        LOGDOG_(logger, error,
            log::message="request to settings failed with non-retriable http status",
            log::http_status=response.status,
            log::response_body=response.body
        );
        return make_unexpected(error_code(Error::nonRetryableStatus));
    }
    LOGDOG_(logger, error,
        log::message="request to settings failed after all retries",
        log::http_status=response.status,
        log::response_body=response.body
    );
    return make_unexpected(error_code(Error::retriesExceeded));
}

expected<bool> SettingsClientImpl::parseGetCollectAddressesFieldResult(const Response& response) const {
    SettingsResponse settingsResponse;
    settingsResponse = yamail::data::deserialization::fromJson<SettingsResponse>(response.body);

    if (!settingsResponse.settings) {
        return make_unexpected(error_code(Error::badResponse, "settings field not found"));
    }
    const auto& singleSettings = settingsResponse.settings->single_settings;

    if (!singleSettings) {
        return make_unexpected(error_code(Error::badResponse, "single settings field not found"));
    }
    const auto& collectAddressesField = singleSettings->collect_addresses;

    if (!collectAddressesField) {
        return make_unexpected(error_code(Error::badResponse, "collect addresses field not found"));
    }

    if (collectAddressesField->empty()) {
        return false;
    }
    return true;
}

} // collie::services::settings
