#include "sheltie_client_impl.hpp"

#include <src/logic/interface/types/carddav_vcard_transforms.hpp>
#include <src/logic/interface/types/reflection/vcard.hpp>

#include <yplatform/encoding/url_encode.h>

#include <yamail/data/deserialization/yajl.h>
#include <yamail/data/serialization/yajl.h>

#include <butil/http/headers.h>

namespace collie::services::sheltie {

namespace {

template <class T>
T fromJson(const TaskContextPtr& context, const std::string& value) {
    using yamail::data::deserialization::fromJson;
    try {
        return fromJson<T>(value);
    } catch (const std::exception& e) {
        logException(context->logger(), e,
            log::message="failed to parse sheltie response",
            log::response_body=value
        );
        throw std::runtime_error("failed to parse sheltie response");
    }
}

}

SheltieClientImpl::SheltieClientImpl(const GetClusterClient& getHttpClient_)
    : getHttpClient(getHttpClient_) {}

MapUriVcardRfc SheltieClientImpl::toVcard(
    const TaskContextPtr& context,
    const Uid& uid,
    const MapUriVcardJson& vcardsAsJson
) const {
    using yamail::data::serialization::toJson;
    using yplatform::url_encode;
    using namespace http_getter;
    const auto request = post("/v1/to_vcard")
                        .headers(requestId=context->requestId(), "Content-Type"_hdr="application/json")
                        .body(toJson(vcardsAsJson).str())
                        .getArgs("uid"_arg=uid)
                        .make();
    const auto response =  performWithRetries(context, request);
    return fromJson<MapUriVcardRfc>(context, response.body);
}

std::string SheltieClientImpl::fromVcard(
    const TaskContextPtr& context,
    const Uid& uid,
    std::string vcardAsRfc
) const {
    using yamail::data::serialization::toJson;
    using yplatform::url_encode;
    using namespace http_getter;
    const auto request = post("/v1/from_vcard")
                        .headers(requestId=context->requestId(), "Content-Type"_hdr="application/text")
                        .body(std::move(vcardAsRfc))
                        .getArgs("uid"_arg=uid)
                        .make();
    const auto response =  performWithRetries(context, request);
    return response.body;
}

SheltieClientImpl::Response SheltieClientImpl::performWithRetries(const TaskContextPtr& context,
        const http_getter::Request& request) const {
    using namespace std::string_literals;

    const auto httpClient = getHttpClient();

    boost::system::error_code ec;
    const auto response = http_getter::asyncRun(*httpClient, context, request, context->yield()[ec]);
    if (ec) {
        throw std::runtime_error("request to sheltie failed");
    }
    if (response.status == 200) {
        return response;
    }
    const auto logger = logdog::bind(context->logger(),
        log::http_status=response.status,
        log::response_body=response.body);
    if (response.status / 100 != 5) {
        LOGDOG_(logger, error,
            log::message="request to sheltie failed with non-retriable http status"
        );
        throw std::runtime_error("request to sheltie failed with http status: "s
            + std::to_string(response.status));
    }
    LOGDOG_(logger, error,
        log::message="request to sheltie failed after all retries"
    );
    throw std::runtime_error("request to sheltie failed after all retries with http status: "s
        + std::to_string(response.status));
}

} // collie::services::sheltie
