#include "staff_client_impl.hpp"
#include "error.hpp"

#include <src/services/retry.hpp>
#include <butil/http/headers.h>

BOOST_FUSION_ADAPT_STRUCT(collie::services::staff::Position,
    ru
)

BOOST_FUSION_ADAPT_STRUCT(collie::services::staff::Official,
    is_dismissed,
    position
)

BOOST_FUSION_ADAPT_STRUCT(collie::services::staff::Personal,
    about,
    birthday
)

BOOST_FUSION_ADAPT_STRUCT(collie::services::staff::NamePart,
    ru
)

BOOST_FUSION_ADAPT_STRUCT(collie::services::staff::Name,
    first,
    middle,
    last
)

BOOST_FUSION_ADAPT_STRUCT(collie::services::staff::Account,
    type,
    value
)

BOOST_FUSION_ADAPT_STRUCT(collie::services::staff::Phone,
    number,
    type
)

BOOST_FUSION_ADAPT_STRUCT(collie::services::staff::Person,
    id,
    name,
    official,
    personal,
    uid,
    work_email,
    work_phone,
    accounts,
    phones
)

BOOST_FUSION_ADAPT_STRUCT(collie::services::staff::Links,
    next
)

BOOST_FUSION_ADAPT_STRUCT(collie::services::staff::PersonsResponse,
    links,
    result
)

namespace collie::services::staff {

namespace {

auto makeRetryCondition(std::size_t retries) {
    return makeCompositeRetryCondition(retry_condition::LimitedRetries{retries},
            retry_condition::RetryHttp5xx{});
}

} // namespace

StaffClientImpl::StaffClientImpl(const Config& config, const GetHttpClient& getHttpClient,
        const GetTvm2Module& getTvm2Module)
        : config(config)
        , getHttpClient(getHttpClient)
        , getTvm2Module(getTvm2Module) {
    std::ostringstream url;
    url << config.location << "/v3/persons?_pretty=1&_limit=" << std::to_string(config.limit) << "&_fields" <<
            "=id,name.first.ru,name.middle,name.last.ru,official.is_dismissed,official.position.ru," <<
            "personal.about,personal.birthday,uid,work_email,work_phone,accounts.type,accounts.value," <<
            "phones.number,phones.type";
    nextPersonsRequestUrl = url.str();
}

expected<Persons> StaffClientImpl::getPersons(const TaskContextPtr& context) {
    if (!nextPersonsRequestUrl) {
        return Persons{};
    }

    if (!tvmServiceTicket) {
        auto ticket{getTvmServiceTicket(context)};
        if (!ticket) {
            return make_unexpected(std::move(ticket).error());
        }

        tvmServiceTicket = std::move(ticket).value();
    }

    return getPersonsImpl(context).bind([&](auto&& response) {
        nextPersonsRequestUrl = std::move(response.links.next);
        return response.result;
    });
}

expected<PersonsResponse> StaffClientImpl::getPersonsImpl(const TaskContextPtr& context) const {
    using namespace http_getter;
    const auto request = get(*nextPersonsRequestUrl)
                        .headers(requestId=context->requestId(), serviceTicket=*tvmServiceTicket)
                        .timeouts(config.httpOptions.timeouts.total, config.httpOptions.timeouts.connect)
                        .make();
    return perform(context, request).bind([&](const auto& response) {
        const auto httpStatusOk{200};
        if (response.status == httpStatusOk) {
            const std::string source{"Staff"};
            return fromJson<PersonsResponse, Error>(response.body, context, source);
        } else {
            return make_expected_from_error<PersonsResponse>(error_code{services::Error::httpError});
        }
    });
}

expected<std::string> StaffClientImpl::getTvmServiceTicket(const TaskContextPtr& context) const {
    std::string result;
    const auto ec{getTvm2Module()->get_service_ticket(config.targetServiceName, result)};
    if (ec) {
        LOGDOG_(context->logger(), error, log::message="failed to get TVM2 service ticket for Staff request",
                log::error_code=ec);
        return make_unexpected(error_code{Error::tvmServiceTicketError, ec.message()});
    }

    return result;
}

expected<yhttp::response> StaffClientImpl::perform(const TaskContextPtr& context,
        const http_getter::Request& request) const {
    auto trap{[](auto&& error) {
        if (error == services::Error::httpError) {
            return make_unexpected(error_code{services::Error::httpError,
                    "request to Staff failed after all retries"});
        } else {
            return make_unexpected(std::move(error));
        }
    }};

    return performWithRetries(*getHttpClient(), context, request, makeRetryCondition(
            config.retries)).catch_error(std::move(trap));
}

} // namespace collie::services::staff
