#pragma once

#include <yamail/data/serialization/libxml.h>

namespace collie::services {

class XmlWriter : public yamail::data::reflection::Visitor {
public:
    using TextWriter = yamail::data::serialization::libxml::TextWriter;

    template <class ... Ts>
    using NamedItemTag = yamail::data::reflection::NamedItemTag<Ts ...>;

    using SequenceItemTag = yamail::data::reflection::SequenceItemTag;

    XmlWriter(TextWriter& writer)
        : writer(writer) {}

    template <class T, class Tag>
    void apply(const T& value, Tag tag) const {
        writer.startDocument();
        applyVisitor(value, *this, tag);
        writer.endDocument();
    }

    template <class Tag>
    void onValue(bool b, Tag tag) const {
        addNode(tag, (b ? "true" : "false"));
    }

    template <class P, class Tag>
    void onValue(const P & p, Tag tag) const {
        addNode(tag, p);
    }

    template <class S, class Tag>
    XmlWriter onStructStart(const S& , Tag tag) const {
        startNode(tag);
        return *this;
    }

    template <class S, class Tag>
    void onStructEnd(const S&, Tag tag) const {
        endNode(tag);
    }

    template <class M, class Tag>
    XmlWriter onMapStart(const M&, Tag tag) const {
        startNode(tag);
        return *this;
    }

    template <class M, class Tag>
    void onMapEnd(const M&, Tag tag) const {
        endNode(tag);
    }

    template <class S>
    XmlWriter onSequenceStart(const S&, SequenceItemTag) {
        name = "value";
        return *this;
    }

    template <class S, class Tag>
    XmlWriter onSequenceStart(const S&, Tag tag) {
        name = yamail::data::reflection::name(tag);
        return *this;
    }

    template <class S, class Tag>
    void onSequenceEnd(const S&, Tag) const {}

private:
    TextWriter& writer;
    const char* name;

    template <class ... T>
    void startNode(NamedItemTag<T...> tag) const {
        writer.startNode(yamail::data::reflection::name(tag));
    }

    void startNode(SequenceItemTag) const {
        writer.startNode(name);
    }

    template <class Tag>
    void startNode(Tag) const {
        writer.startNode("value");
    }

    template <class Tag>
    void endNode(Tag) const {
        writer.endNode();
    }

    template <class Tag, class T>
    void addNode(Tag tag, const T& value) const {
        startNode(tag);
        writer.writeValue(value);
        endNode(tag);
    }
};

template <typename T>
inline yamail::data::serialization::libxml::Buffer toXml(const T& v, const std::string& name) {
    using namespace yamail::data::serialization::libxml;
    auto buffer = createBuffer();
    TextWriter textWriter(buffer);
    XmlWriter(textWriter).apply(v, namedItemTag(name));
    return buffer;
}

} // namespace namespace collie::services
