#pragma once

#include <src/expected.hpp>
#include <src/services/db/events_queue/query.hpp>
#include <src/services/db/execute.hpp>
#include <src/services/db/request.hpp>
#include <src/services/db/utils.hpp>

namespace collie::sync::common {

template<typename Connection> class ServiceSync {
public:
    using ServiceType = services::db::events_queue::ServiceType;
    ServiceSync(Connection& connection, ServiceType serviceType)
        : connection(connection)
        , serviceType(std::move(serviceType)) {
    }

    expected<std::optional<std::time_t>> getTimestamp() {
        services::db::events_queue::query::GetServiceSyncTimestamp query;
        query.service_type = serviceType;
        return services::db::request(connection, std::as_const(query)).bind([&](auto&& row) {
            using Result = std::optional<std::time_t>;
            if (row.empty()) {
                LOGDOG_(services::db::unwrap(connection).context()->logger(), notice,
                        log::query=ozo::get_query_name(query), log::message="locked");
                return Result{};
            }

            const auto result{services::db::expectSingleRow(std::move(row))};
            LOGDOG_(services::db::unwrap(connection).context()->logger(), notice,
                    log::query=ozo::get_query_name(query),
                    log::sync_timestamp=result.sync_timestamp);

            return Result{result.sync_timestamp};
        });
    }

    expected<void> setTimestamp() {
        services::db::events_queue::query::SetServiceSyncTimestamp query;
        query.service_type = serviceType;
        return services::db::execute(connection, std::as_const(query)).bind([&] {
            LOGDOG_(services::db::unwrap(connection).context()->logger(), notice,
                    log::query=ozo::get_query_name(query));
        });
    }

private:
    Connection& connection;
    ServiceType serviceType;
};

} // namespace collie::sync::common
