#pragma once

#include "id_map.hpp"

#include <src/logic/db/contacts/create_contacts.hpp>
#include <src/logic/db/contacts/remove_contacts_completely.hpp>
#include <src/logic/db/contacts/update_contacts.hpp>

namespace collie::sync::common {

struct UpdateContactsData {
    std::vector<logic::ContactId> contactIdsToDelete;
    std::vector<logic::UpdatedContact> contactsToUpdate;
    std::vector<logic::NewContact> contactsToCreate;
};

struct UpdateIdMapData {
    std::vector<KeyId> keyIdsToDelete;
    std::vector<KeyId> keyIdsToAdd;
};

template<typename EventsQueueTransaction, typename ContactsTransaction, typename AddIdMapElements,
        typename DeleteByKeyIds> class UpdateDb {
public:
    UpdateDb(EventsQueueTransaction& eventsQueueTransaction, ContactsTransaction& contactsTransaction,
            std::size_t chunkSizeForModifyingOp)
            : eventsQueueTransaction(eventsQueueTransaction)
            , contactsTransaction(contactsTransaction)
            , chunkSizeForModifyingOp(chunkSizeForModifyingOp) {
    }

    expected<std::vector<logic::ContactId>> operator()(UpdateContactsData data) {
        auto deleteContacts{[&]{return logic::db::contacts::removeContactsCompletely(contactsTransaction,
                data.contactIdsToDelete, chunkSizeForModifyingOp).bind([](auto&&){});}};
        auto updateContacts{[&]{return logic::db::contacts::updateContacts({std::move(data.contactsToUpdate),
                {}}, contactsTransaction, chunkSizeForModifyingOp).bind([](auto&&){});}};
        auto createContacts{[&]{return logic::db::contacts::createContactsAndReturnIds(std::move(
                data.contactsToCreate), contactsTransaction, chunkSizeForModifyingOp);}};
        return deleteContacts().bind(std::move(updateContacts)).bind(std::move(createContacts));
    }

    expected<void> operator()(UpdateIdMapData data, std::vector<logic::ContactId> contactIds) {
        auto deleteIdMapElements{[&]{
            IdMap idMap{eventsQueueTransaction};
            return idMap.template deleteByKeyIds<DeleteByKeyIds>(std::move(data.keyIdsToDelete));
        }};

        auto addIdMapElements{[&]{
            std::vector<IdMapElement> elements;
            std::transform(data.keyIdsToAdd.begin(), data.keyIdsToAdd.end(), contactIds.begin(),
                    std::back_inserter(elements), [](auto keyId, auto valueId){return IdMapElement{
                            keyId, valueId};});
            IdMap idMap{eventsQueueTransaction};
            return idMap.template addElements<AddIdMapElements>(std::move(elements));
        }};

        return deleteIdMapElements().bind(std::move(addIdMapElements));
    }

private:
    EventsQueueTransaction& eventsQueueTransaction;
    ContactsTransaction& contactsTransaction;
    std::size_t chunkSizeForModifyingOp;
};

} // namespace collie::sync::common
