#include "utils.hpp"

namespace collie::services::db::events_queue {

inline bool operator<(const IdMapElement& idMapElement, ValueId valueId) {
    return idMapElement.value_id < valueId;
}

inline bool operator<(ValueId valueId, const IdMapElement& idMapElement) {
    return valueId < idMapElement.value_id;
}

}

namespace collie::sync::common {

std::vector<KeyId> getKeyIdsFromIdMapElements(const std::vector<IdMapElement>& idMapElements) {
    std::vector<KeyId> keyIds;
    keyIds.reserve(idMapElements.size());
    std::transform(idMapElements.begin(), idMapElements.end(), std::back_inserter(keyIds),
            [](const auto& element){return element.key_id;});
    return keyIds;
}

std::vector<ValueId> getValueIdsFromIdMapElements(const std::vector<IdMapElement>& idMapElements) {
    std::vector<ValueId> valueIds;
    valueIds.reserve(idMapElements.size());
    std::transform(idMapElements.begin(), idMapElements.end(), std::back_inserter(valueIds),
            [](const auto& element){return element.value_id;});
    return valueIds;
}

ContactsInfoIdMap makeContactInfoIdMap(ContactsInfo& contactsInfo) {
    ContactsInfoIdMap map;
    for (const auto& element : contactsInfo.contacts) {
        map.emplace(element.contact_id, element);
    }

    return map;
}

ContactIdSearchMap makeContactIdSearchMap(
        const std::vector<IdMapElement>& idMapElements,
        ContactsInfoIdMap& idMapContactsInfo) {
    ContactIdSearchMap map;
    for (const auto& element : idMapElements) {
        map.emplace(element.key_id, idMapContactsInfo[element.value_id]);
    }

    return map;
}

std::vector<IdMapElement> getDbCorrectionData(
        std::vector<ValueId> contactIds,
        std::vector<IdMapElement> idMapElements,
        std::vector<ValueId>& contactIdsToDelete,
        std::vector<KeyId>& keyIdsToDelete) {
    if (!std::is_sorted(contactIds.begin(), contactIds.end())) {
        std::sort(contactIds.begin(), contactIds.end());
    }

    auto compareIdMapElements{[](const auto& left, const auto& right){
            return left.value_id < right.value_id;}};
    if (!std::is_sorted(idMapElements.begin(), idMapElements.end(), compareIdMapElements)) {
        std::sort(idMapElements.begin(), idMapElements.end(), std::move(compareIdMapElements));
    }

    std::set_difference(contactIds.begin(), contactIds.end(), idMapElements.begin(), idMapElements.end(),
            std::back_inserter(contactIdsToDelete));

    std::vector<IdMapElement> idMapElementsToDelete;
    std::set_difference(idMapElements.begin(), idMapElements.end(), contactIds.begin(), contactIds.end(),
            std::back_inserter(idMapElementsToDelete));
    keyIdsToDelete = getKeyIdsFromIdMapElements(idMapElementsToDelete);

    std::vector<IdMapElement> actualIdMapElements;
    std::set_intersection(
            std::make_move_iterator(idMapElements.begin()),
            std::make_move_iterator(idMapElements.end()),
            contactIds.begin(),
            contactIds.end(),
            std::back_inserter(actualIdMapElements));

    return actualIdMapElements;
}

} // namespace collie::sync::common
