#pragma once

#include "error.hpp"

#include <src/logic/db/contacts/get_contact_ids_by_list_ids.hpp>
#include <src/logic/db/contacts/get_contacts.hpp>
#include <src/logic/db/contacts/get_shared_lists.hpp>
#include <src/logic/db/contacts/subscribe_to_contacts.hpp>
#include <src/logic/db/contacts/unsubscribe_from_contacts.hpp>
#include <src/services/db/begin.hpp>
#include <src/services/db/commit.hpp>
#include <src/services/db/get_connection.hpp>
#include <src/services/db/org_user_id.hpp>
#include <src/services/db/passport_user_id.hpp>
#include <src/services/ml/ml_client.hpp>
#include <src/services/staff/staff_client.hpp>
#include <src/sync/common/service_sync.hpp>
#include <src/sync/common/update_db.hpp>
#include <src/sync/common/utils.hpp>

#include <boost/algorithm/string.hpp>

namespace collie::services::db::events_queue {

inline bool operator<(const IdMapElement& idMapElement, const services::ml::MlEntry& mlEntry)
{
    return idMapElement.key_id < mlEntry.id;
}

}

namespace collie::services::ml {

inline bool operator<(const MlEntry& mlEntry, const sync::common::IdMapElement& idMapElement)
{
    return mlEntry.id < idMapElement.key_id;
}

}

namespace collie::sync::ml {

template<typename MakeContactsConnectionProvider, typename MakeEventsQueueConnectionProvider> class Worker {
public:
    using MlClientPtr = services::ml::MlClientPtr;
    Worker(
            const common::WorkerContextPtr& workerContext,
            const MakeContactsConnectionProvider& makeContactsConnectionProvider,
            const MakeEventsQueueConnectionProvider& makeEventsQueueConnectionProvider,
            const MlClientPtr& mlClient,
            const Config& config)
            : workerContext(workerContext)
            , makeContactsConnectionProvider(makeContactsConnectionProvider)
            , makeEventsQueueConnectionProvider(makeEventsQueueConnectionProvider)
            , ml(mlClient)
            , config(config) {
    }

    void operator()() const {
        const auto eventsQueueProvider{makeEventsQueueConnectionProvider(workerContext->getTaskContext())};
        auto transact{[&](auto&& eventsQueueTransaction) {
            auto syncRequired{[&](auto&& syncTimestamp) {
                return syncRequiredImpl(std::move(syncTimestamp));
            }};

            auto sync{[&]{return syncImpl(eventsQueueTransaction);}};

            using ServiceType = services::db::events_queue::ServiceType;
            const std::string serviceType{"ml"};
            common::ServiceSync serviceSync{eventsQueueTransaction, ServiceType{serviceType}};

            auto setTimestamp{[&]{return serviceSync.setTimestamp();}};
            auto returnTransaction{[&]{return std::move(eventsQueueTransaction);}};
            return serviceSync.getTimestamp().bind(std::move(syncRequired)).bind(std::move(sync)).bind(
                    std::move(setTimestamp)).bind(std::move(returnTransaction));
        }};

        auto commit{[&](auto&& eventsQueueTransaction){return services::db::commit(eventsQueueProvider,
                std::move(eventsQueueTransaction));}};
        auto handleError{[&](auto&& errorCode){LOGDOG_(workerContext->logger(), error,
                log::error_code=std::move(errorCode), log::message="ML synchronization failed");}};

        services::db::begin(eventsQueueProvider).bind(std::move(transact)).bind(std::move(commit)).
                catch_error(std::move(handleError));
    }

private:
    expected<void> syncRequiredImpl(std::optional<std::time_t> syncTimestamp) const {
        if (!syncTimestamp) {
            return make_expected_from_error<void>(error_code{Error::syncIsOngoing});
        }

        if (std::chrono::system_clock::now() < (std::chrono::system_clock::time_point{std::chrono::
                seconds{*syncTimestamp}} + std::chrono::minutes{config.syncIntervalInMinutes})) {
            return make_expected_from_error<void>(error_code{Error::syncCompletedByPeer});
        }

        return {};
    }

    template<typename EventsQueueTransaction> expected<void> syncImpl(
            EventsQueueTransaction& eventsQueueTransaction) const {
        const auto contactsProvider{makeContactsConnectionProvider(workerContext->getTaskContext(),
                services::db::OrgUserId{config.orgId})};
        auto withOrgListId{[&](auto orgListId) {
            common::UpdateContactsData updateContactsData;
            common::UpdateIdMapData updateIdMapData;
            auto prepareData{[&](auto&& idMapElements){return prepareDataImpl(contactsProvider, orgListId,
                    std::move(idMapElements), updateContactsData, updateIdMapData);}};
            auto processData{[&]{return processDataImpl(eventsQueueTransaction, contactsProvider,
                    std::move(updateContactsData), std::move(updateIdMapData), orgListId);}};
            common::IdMap idMap{eventsQueueTransaction};
            using GetMlIdMapElements = services::db::events_queue::query::GetMlIdMapElements;
            return idMap.template getElements<GetMlIdMapElements>().bind(std::move(prepareData)).bind(
                    std::move(processData));
        }};

        return logic::db::contacts::getDefaultListId(contactsProvider).bind(std::move(withOrgListId));
    }

    logic::Vcard makeVcard(services::ml::MlEntry mlEntry) const {
        logic::Vcard vcard;
        if (!mlEntry.email.empty()) {
            logic::Email email;
            email.email = mlEntry.email;
            vcard.emails = std::vector<logic::Email>{std::move(email)};
        }

        logic::Name name;
        if (mlEntry.name) {
            name.first = std::move(mlEntry.name);
        } else {
            std::vector<std::string> tokens;
            const std::string separator{"@"};
            boost::algorithm::split(tokens, mlEntry.email, boost::algorithm::is_any_of(separator),
                    boost::algorithm::token_compress_on);
            name.first = std::move(tokens[0]);
        }

        *name.first += " (рассылка)";
        vcard.names = std::vector<logic::Name>{std::move(name)};

        return vcard;
    }

    logic::NewContact makeNewContact(services::ml::MlEntry mlEntry) const {
        logic::NewContact contact;
        contact.vcard = makeVcard(std::move(mlEntry));
        return contact;
    }

    logic::UpdatedContact makeUpdatedContact(logic::ContactId contactId,
            services::ml::MlEntry mlEntry) const {
        logic::UpdatedContact contact;
        contact.contact_id = contactId;
        contact.vcard = makeVcard(std::move(mlEntry));
        return contact;
    }

    bool isExistingContactEqMlContact(const logic::Vcard& existingContactInfo, const logic::Vcard& mlContactInfo) const {
        return existingContactInfo.names == mlContactInfo.names && existingContactInfo.emails == mlContactInfo.emails;
    }

    void processMlEntry(
            services::ml::MlEntry mlEntry,
            const common::ContactIdSearchMap& searchMap,
            std::vector<logic::NewContact>& contactsToCreate,
            std::vector<logic::UpdatedContact>& contactsToUpdate,
            std::vector<common::KeyId>& keyIdsToAdd) const {
        const auto element{searchMap.find(mlEntry.id)};
        if (element == searchMap.end()) {
            keyIdsToAdd.emplace_back(mlEntry.id);
            contactsToCreate.emplace_back(makeNewContact(std::move(mlEntry)));
        } else {
            auto mlContactInfo = makeVcard(mlEntry);
            if (!isExistingContactEqMlContact(element->second.vcard, mlContactInfo)) {
                contactsToUpdate.emplace_back(makeUpdatedContact(element->second.contact_id, std::move(mlEntry)));
            }
        }
    }

    void prepareIdsToDelete(
            std::vector<common::IdMapElement> idMapElements,
            services::ml::MlEntries mlEntries,
            std::vector<logic::ContactId>& contactIdsToDelete,
            std::vector<common::KeyId>& keyIdsToDelete) const {
        auto compareIdMapElements{[](const auto& left, const auto& right){
                return left.key_id < right.key_id;}};
        std::sort(idMapElements.begin(), idMapElements.end(), std::move(compareIdMapElements));

        auto compareMlEntries{[](const auto& left, const auto& right){return left.id < right.id;}};
        if (!std::is_sorted(mlEntries.begin(), mlEntries.end(), compareMlEntries)) {
            std::sort(mlEntries.begin(), mlEntries.end(), std::move(compareMlEntries));
        }

        auto equalMlEntries{[](const auto& left, const auto& right){return left.id == right.id;}};
        mlEntries.erase(std::unique(mlEntries.begin(), mlEntries.end(), std::move(equalMlEntries)),
                mlEntries.end());

        std::vector<common::IdMapElement> idMapElementsToDelete;
        std::set_difference(idMapElements.begin(), idMapElements.end(), mlEntries.begin(), mlEntries.end(),
                std::back_inserter(idMapElementsToDelete));

        const auto newKeyIdsToDelete{common::getKeyIdsFromIdMapElements(idMapElementsToDelete)};
        keyIdsToDelete.insert(keyIdsToDelete.end(), newKeyIdsToDelete.begin(), newKeyIdsToDelete.end());

        const auto newContactIdsToDelete{common::getValueIdsFromIdMapElements(idMapElementsToDelete)};
        contactIdsToDelete.insert(contactIdsToDelete.end(), newContactIdsToDelete.begin(),
                newContactIdsToDelete.end());
    }

    expected<void> addNewMlDataImpl(
            std::vector<common::IdMapElement> idMapElements,
            common::ContactsInfo& contactsInfo,
            common::UpdateContactsData& updateContactsData,
            common::UpdateIdMapData& updateIdMapData) const {
        auto mlEntries{ml->getMlEntries(workerContext->getTaskContext())};
        if (!mlEntries) {
            return make_expected_from_error<void>(std::move(mlEntries).error());
        }

        std::unordered_set<common::KeyId> receivedKeyIds;
        const auto duplicate{[&](const auto& mlEntry){
                return receivedKeyIds.find(mlEntry.id) != receivedKeyIds.end();}};
        const auto logDuplicate{[&](auto id){LOGDOG_(workerContext->logger(), warning,
                log::message="duplicated ID received from ML", log::ml_id=id);}};

        auto contactInfoIdMap{common::makeContactInfoIdMap(contactsInfo)};
        const auto searchMap{common::makeContactIdSearchMap(idMapElements, contactInfoIdMap)};
        for (auto& mlEntry : mlEntries.value()) {
            if (duplicate(mlEntry)) {
                logDuplicate(mlEntry.id);
            } else {
                receivedKeyIds.emplace(mlEntry.id);
                processMlEntry(mlEntry, searchMap, updateContactsData.contactsToCreate,
                        updateContactsData.contactsToUpdate, updateIdMapData.keyIdsToAdd);
            }
        }

        prepareIdsToDelete(std::move(idMapElements), std::move(mlEntries).value(),
                updateContactsData.contactIdsToDelete, updateIdMapData.keyIdsToDelete);

        return {};
    }

    template<typename ContactsProvider> expected<void> prepareDataImpl(
            ContactsProvider& contactsProvider,
            logic::ListId orgListId,
            std::vector<common::IdMapElement> idMapElements,
            common::UpdateContactsData& updateContactsData,
            common::UpdateIdMapData& updateIdMapData) const {
        auto getDbCorrectionData{[&](auto&& contactIds) {
            return common::getDbCorrectionData(std::move(contactIds), std::move(idMapElements),
                    updateContactsData.contactIdsToDelete, updateIdMapData.keyIdsToDelete);
        }};

        auto addNewData{[&](auto&& actualIdMapElements) {
            auto addNewMlData{[&](auto&& contactsInfo){return addNewMlDataImpl(std::move(
                actualIdMapElements), contactsInfo, updateContactsData, updateIdMapData);}};
            
            auto contactIds{common::getKeyIdsFromIdMapElements(idMapElements)};
            using services::db::contacts::query::GetContacts;
            return logic::db::contacts::getContacts<GetContacts>(contactsProvider, contactIds,
                    {orgListId}, {},{}).bind(std::move(addNewMlData));
        }};

        return logic::db::contacts::getContactIdsByListIds(contactsProvider, {orgListId}, {}).bind(std::move(
                getDbCorrectionData)).bind(std::move(addNewData));
    }

    template<typename EventsQueueTransaction, typename ContactsProvider> expected<void> processDataImpl(
            EventsQueueTransaction& eventsQueueTransaction,
            ContactsProvider& contactsProvider,
            common::UpdateContactsData updateContactsData,
            common::UpdateIdMapData updateIdMapData,
            logic::ListId orgListId) const {
        auto transact{[&](auto&& contactsTransaction) {
            auto updateDb{[&]{return updateDbImpl(eventsQueueTransaction, contactsTransaction,
                    std::move(updateContactsData), std::move(updateIdMapData));}};
            auto returnTransaction{[&]{return std::move(contactsTransaction);}};
            return updateDb().bind(std::move(returnTransaction));
        }};

        auto commit{[&](auto&& contactsTransaction){return services::db::commit(contactsProvider,
                std::move(contactsTransaction));}};
        auto ignoreResult{[](auto&&){}};

        auto shareLists{[&]{return shareListsImpl(contactsProvider, orgListId);}};

        return services::db::begin(contactsProvider).bind(std::move(transact)).bind(std::move(commit)).bind(
                std::move(ignoreResult)).bind(std::move(shareLists));
    }

    template<typename EventsQueueTransaction, typename ContactsTransaction> expected<void> updateDbImpl(
            EventsQueueTransaction& eventsQueueTransaction,
            ContactsTransaction& contactsTransaction,
            common::UpdateContactsData updateContactsData,
            common::UpdateIdMapData updateIdMapData) const {
        using AddMlIdMapElements = services::db::events_queue::query::AddMlIdMapElements;
        using DeleteContactIdsByMlIds = services::db::events_queue::query::DeleteContactIdsByMlIds;
        common::UpdateDb<EventsQueueTransaction, ContactsTransaction, AddMlIdMapElements,
                DeleteContactIdsByMlIds> updateDb{eventsQueueTransaction, contactsTransaction,
                        config.chunkSizeForModifyingOp};
        auto updateIdMap{[&](auto&& contactIds){return updateDb(std::move(updateIdMapData), std::move(
                contactIds));}};
        return updateDb(std::move(updateContactsData)).bind(std::move(updateIdMap));
    }

    using Uids = std::vector<services::staff::Uid>;
    template<typename ContactsProvider> expected<void> shareListsImpl(ContactsProvider&& contactsProvider,
            logic::ListId orgListId) const {
        const auto getUserIds{[](auto&& provider, auto listId) {
            const auto processSharedLists{[&](const auto& sharedLists) {
                auto filter{[&](const auto& list){return list.list_id == listId;}};
                auto listTransformer{[](const auto& list){return list.client_user_id;}};
                Uids userIds;
                boost::copy(sharedLists | boost::adaptors::filtered(std::move(filter)) | boost::adaptors::
                        transformed(std::move(listTransformer)), std::back_inserter(userIds));
                if (!std::is_sorted(userIds.begin(), userIds.end())) {
                    std::sort(userIds.begin(), userIds.end());
                }

                return userIds;
            }};

            return logic::db::contacts::getSharedLists(provider).bind(processSharedLists);
        }};

        auto existingUserIds{getUserIds(contactsProvider, orgListId)};
        if (!existingUserIds) {
            return make_unexpected(std::move(existingUserIds).error());
        }

        auto staffProvider{makeContactsConnectionProvider(workerContext->getTaskContext(),
                services::db::OrgUserId{config.staffId})};
        auto staffListId{logic::db::contacts::getDefaultListId(staffProvider)};
        if (!staffListId) {
            return make_unexpected(std::move(staffListId).error());
        }

        auto actualUserIds{getUserIds(staffProvider, *staffListId)};
        if (!actualUserIds) {
            return make_unexpected(std::move(actualUserIds).error());
        }

        Uids userIdsToUnsubscribe;
        std::set_difference(existingUserIds->begin(), existingUserIds->end(), actualUserIds->begin(),
                actualUserIds->end(), std::back_inserter(userIdsToUnsubscribe));

        Uids userIdsToSubscribe;
        std::set_difference(actualUserIds->begin(), actualUserIds->end(), existingUserIds->begin(),
                existingUserIds->end(), std::back_inserter(userIdsToSubscribe));

        auto unsubscribe{[&](auto&& userProvider){return logic::db::contacts::unsubscribeFromContacts(
                contactsProvider, std::move(userProvider), orgListId);}};
        manageSubscriptions(std::move(userIdsToUnsubscribe), std::move(unsubscribe));

        const std::string clientListName{"ml_contacts"};
        auto subscribe{[&](auto&& userProvider){return logic::db::contacts::subscribeToContacts(
                contactsProvider, std::move(userProvider), orgListId, clientListName);}};
        manageSubscriptions(std::move(userIdsToSubscribe), std::move(subscribe));

        return make_expected();
    }

    template<typename Handler> void manageSubscriptions(Uids ids, Handler handler) const {
        for (auto id : ids) {
            if (workerContext->mustStop()) {
                LOGDOG_(workerContext->logger(), error, log::error_code=error_code{Error::syncInterrupted},
                        log::message="ML subscription managing failed");
                return;
            }

            auto provider{makeContactsConnectionProvider(workerContext->getTaskContext(),
                    services::db::PassportUserId{id})};
            auto connection{getConnection(provider)};
            if (connection) {
                auto result{handler(std::move(provider))};
                if (!result) {
                    LOGDOG_(workerContext->logger(), error, log::error_code=std::move(result).error(),
                            log::user_id=id, log::message="Handler error while subscription managing");
                }
            } else {
                LOGDOG_(workerContext->logger(), warning, log::error_code=std::move(connection).error(),
                        log::user_id=id, log::message="connection getting error while subscription managing");
            }
        }
    }

    common::WorkerContextPtr workerContext;
    const MakeContactsConnectionProvider& makeContactsConnectionProvider;
    const MakeEventsQueueConnectionProvider& makeEventsQueueConnectionProvider;
    const MlClientPtr& ml;
    const Config& config;
};

} // namespace collie::sync::ml
