#pragma once

#include <src/error_code.hpp>

namespace collie::sync::staff {

enum class Error {
    ok,
    syncIsOngoing,
    syncCompletedByPeer,
    syncInterrupted
};

} // namespace collie::sync::staff

namespace boost::system {

template<> struct is_error_code_enum<collie::sync::staff::Error> : std::true_type {};

} // namespace boost::system

namespace collie::sync::staff {

class ErrorCategory final : public boost::system::error_category {
public:
    const char* name() const noexcept override {
        return "collie::sync::staff::ErrorCategory";
    }

    std::string message(int value) const override {
        switch (static_cast<Error>(value)) {
            case Error::ok:
                return "ok";
            case Error::syncIsOngoing:
                return "synchronization is ongoing";
            case Error::syncCompletedByPeer:
                return "synchronization is completed by a peer";
            case Error::syncInterrupted:
                return "synchronization is interrupted";
        }

        return "unknown error code: " + std::to_string(value);
    }

    static const ErrorCategory& instance() {
        static const ErrorCategory errorCategory;
        return errorCategory;
    }
};

inline error_code::base_type make_error_code(Error ec) {
    return error_code::base_type(static_cast<int>(ec), ErrorCategory::instance());
}

} // namespace collie::sync::staff
