#include "module.hpp"
#include "config_reflection.hpp"
#include "run_worker.hpp"

#include <pa/async.h>
#include <src/services/staff/staff_client_impl.hpp>
#include <yamail/data/deserialization/ptree.h>
#include <yplatform/module_registration.h>

namespace collie::sync::staff {

Module::Module(boost::asio::io_context& io, const yplatform::ptree& data)
        : io(io)
        , config(yamail::data::deserialization::fromPtree<Config>(data))
        , workerContext(std::make_shared<common::WorkerContext>())
        , timer(io) {
    if (!pa::async_profiler::is_init()) {
        const auto maxSize{1000000};
        const auto dumpSize{16000};
        pa::async_profiler::init(maxSize, dumpSize, config.profilerLogName);
    }
}

void Module::start() {
    guard = std::make_unique<boost::asio::executor_work_guard<boost::asio::io_context::executor_type>>(
            boost::asio::make_work_guard(io.get()));
    syncTime = boost::posix_time::second_clock::universal_time();
    if (!config.syncAtStartup) {
        syncTime += boost::posix_time::minutes{config.syncIntervalInMinutes};
    }

    scheduleSync();
}

void Module::stop() {
    timer.cancel();
    workerContext->stop();
    guard.reset();
}

void Module::scheduleSync() {
    timer.expires_at(syncTime);
    timer.async_wait([&](const auto& error) {
        if (!error) {
            sync();
            syncTime += boost::posix_time::minutes{config.syncIntervalInMinutes};
            scheduleSync();
        }
    });
}

void Module::sync() {
    std::lock_guard<std::recursive_mutex> lock{mutex};
    if (syncIsOngoing) {
        return;
    }

    syncIsOngoing = true;

    using MakeContactsConnectionProvider = services::db::contacts::MakeConnectionProvider<
            services::db::contacts::WithCheckIsUserExists>;
    using MakeEventsQueueConnectionProvider = services::db::events_queue::MakeConnectionProvider;
    using StaffClient = services::staff::StaffClientImpl;
    const auto getHttpClient{services::makeGetHttpClient(config.services.staff.httpClientModule)};
    const auto getTvm2Module{services::makeGetTvm2Module(config.tvmModule)};
    const RunWorker runWorker{
            workerContext,
            MakeContactsConnectionProvider{config.services.db.contacts, getHttpClient},
            MakeEventsQueueConnectionProvider{config.services.db.eventsQueue, getHttpClient},
            std::make_shared<StaffClient>(config.services.staff, getHttpClient, getTvm2Module)};
    auto coroutineFunction{[=](auto yield) {
        runWorker(std::move(yield), config);
        syncIsOngoing = false;
    }};

    boost::asio::spawn(io.get(), std::move(coroutineFunction), boost::coroutines::attributes(
            config.coroutineStackSize));
}

} // namespace collie::sync::staff

DEFINE_SERVICE_OBJECT(collie::sync::staff::Module)
