#pragma once

#include "config.hpp"
#include "error.hpp"

#include <src/logic/db/contacts/get_contact_ids_by_list_ids.hpp>
#include <src/logic/db/contacts/get_contacts.hpp>
#include <src/logic/db/contacts/get_shared_lists.hpp>
#include <src/logic/db/contacts/subscribe_to_contacts.hpp>
#include <src/logic/db/contacts/unsubscribe_from_contacts.hpp>
#include <src/services/db/begin.hpp>
#include <src/services/db/commit.hpp>
#include <src/services/db/get_connection.hpp>
#include <src/services/db/org_user_id.hpp>
#include <src/services/db/passport_user_id.hpp>
#include <src/services/staff/staff_client.hpp>
#include <src/sync/common/service_sync.hpp>
#include <src/sync/common/update_db.hpp>
#include <src/sync/common/utils.hpp>
#include <src/utils.hpp>

#include <yplatform/util/split.h>

#include <boost/range/adaptor/filtered.hpp>
#include <boost/range/adaptor/transformed.hpp>
#include <boost/range/algorithm/copy.hpp>

namespace collie::sync::staff {

template<typename MakeContactsConnectionProvider, typename MakeEventsQueueConnectionProvider> class Worker {
public:
    using StaffClientPtr = services::staff::StaffClientPtr;
    Worker(
            const common::WorkerContextPtr& workerContext,
            const MakeContactsConnectionProvider& makeContactsConnectionProvider,
            const MakeEventsQueueConnectionProvider& makeEventsQueueConnectionProvider,
            const StaffClientPtr& staffClient,
            const Config& config)
            : workerContext(workerContext)
            , makeContactsConnectionProvider(makeContactsConnectionProvider)
            , makeEventsQueueConnectionProvider(makeEventsQueueConnectionProvider)
            , staff(staffClient)
            , config(config)
            , instantMessengers(yplatform::util::split_unique(config.services.staff.instantMessengers, ","))
            , socialProfiles(yplatform::util::split_unique(config.services.staff.socialProfiles, ",")) {
    }

    void operator()() const {
        const auto eventsQueueProvider{makeEventsQueueConnectionProvider(workerContext->getTaskContext())};
        auto transact{[&](auto&& eventsQueueTransaction) {
            auto syncRequired{[&](auto&& syncTimestamp) {
                return syncRequiredImpl(std::move(syncTimestamp));
            }};

            auto sync{[&]{return syncImpl(eventsQueueTransaction);}};

            using ServiceType = services::db::events_queue::ServiceType;
            const std::string serviceType{"staff"};
            common::ServiceSync serviceSync{eventsQueueTransaction, ServiceType{serviceType}};

            auto setTimestamp{[&]{return serviceSync.setTimestamp();}};
            auto returnTransaction{[&]{return std::move(eventsQueueTransaction);}};
            return serviceSync.getTimestamp().bind(std::move(syncRequired)).bind(std::move(sync)).bind(
                    std::move(setTimestamp)).bind(std::move(returnTransaction));
        }};

        auto commit{[&](auto&& eventsQueueTransaction){return services::db::commit(eventsQueueProvider,
                std::move(eventsQueueTransaction));}};
        auto handleError{[&](auto&& errorCode){LOGDOG_(workerContext->logger(), error,
                log::error_code=std::move(errorCode), log::message="Staff synchronization failed");}};

        services::db::begin(eventsQueueProvider).bind(std::move(transact)).bind(std::move(commit)).
                catch_error(std::move(handleError));
    }

private:
    expected<void> syncRequiredImpl(std::optional<std::time_t> syncTimestamp) const {
        if (!syncTimestamp) {
            return make_expected_from_error<void>(error_code{Error::syncIsOngoing});
        }

        if (std::chrono::system_clock::now() < (std::chrono::system_clock::time_point{std::chrono::
                seconds{*syncTimestamp}} + std::chrono::minutes{config.syncIntervalInMinutes})) {
            return make_expected_from_error<void>(error_code{Error::syncCompletedByPeer});
        }

        return {};
    }

    template<typename EventsQueueTransaction> expected<void> syncImpl(
            EventsQueueTransaction& eventsQueueTransaction) const {
        const auto contactsProvider{makeContactsConnectionProvider(workerContext->getTaskContext(),
                services::db::OrgUserId{config.orgId})};
        auto withOrgListId{[&](auto orgListId) {
            common::UpdateContactsData updateContactsData;
            common::UpdateIdMapData updateIdMapData;
            auto prepareData{[&](auto&& idMapElements){return prepareDataImpl(contactsProvider, orgListId,
                    std::move(idMapElements), updateContactsData, updateIdMapData);}};
            auto processData{[&](auto&& uids){return processDataImpl(eventsQueueTransaction, contactsProvider,
                    std::move(updateContactsData), std::move(updateIdMapData), orgListId, std::move(uids));}};
            common::IdMap idMap{eventsQueueTransaction};
            using GetStaffIdMapElements = services::db::events_queue::query::GetStaffIdMapElements;
            return idMap.template getElements<GetStaffIdMapElements>().bind(std::move(prepareData)).bind(
                    std::move(processData));
        }};

        return logic::db::contacts::getDefaultListId(contactsProvider).bind(std::move(withOrgListId));
    }

    void preprocessAccounts(std::optional<services::staff::Accounts>& accounts) const {
        auto filter{[](const auto& account){return (!account.type) || account.type->empty() ||
                (!account.value) || account.value->empty();}};
        preprocessRange(accounts, std::move(filter));
    }

    void preprocessPhones(std::optional<services::staff::Phones>& phones) const {
        auto filter{[](const auto& phone){return (!phone.number) || phone.number->empty();}};
        preprocessRange(phones, std::move(filter));
    }

    void preprocessPerson(services::staff::Person& person) const {
        preprocessAccounts(person.accounts);
        preprocessPhones(person.phones);
    }

    std::vector<logic::Name> makeNames(services::staff::Name name) const {
        logic::Name result;
        result.first = std::move(name.first.ru);
        result.middle = std::move(name.middle);
        result.last = std::move(name.last.ru);
        return {std::move(result)};
    }

    std::optional<std::vector<logic::Email>> makeEmails(std::optional<std::string> workEmail) const {
        std::optional<std::vector<logic::Email>> result;
        if (workEmail && (!workEmail->empty())) {
            logic::Email email;
            email.email = std::move(workEmail);
            result = std::vector<logic::Email>{std::move(email)};
        }

        return result;
    }

    template<typename Element, typename Transformer> std::optional<std::vector<Element>> makeFromAccounts(
            const std::vector<std::string>& pattern,
            const std::optional<services::staff::Accounts>& accounts,
            Transformer transformer) const {
        std::optional<std::vector<Element>> result;
        if (!accounts) {
            return result;
        }

        auto filter{[&](auto& account){return isInRange(pattern, *account.type);}};
        std::vector<Element> elements;
        boost::copy(*accounts | boost::adaptors::filtered(std::move(filter)) | boost::adaptors::transformed(
                std::move(transformer)), std::back_inserter(elements));
        if (!elements.empty()) {
            result = std::move(elements);
        }

        return result;
    }

    std::optional<std::vector<logic::InstantMessenger>> makeInstantMessengers(
            const std::optional<services::staff::Accounts>& accounts) const {
        auto transformer{[](auto& account) {
            logic::InstantMessenger messenger;
            messenger.service_id = *account.value;
            messenger.service_type = *account.type;
            return messenger;
        }};

        return makeFromAccounts<logic::InstantMessenger>(instantMessengers, accounts, std::move(transformer));
    }

    std::optional<std::vector<logic::SocialProfile>> makeSocialProfiles(
            const std::optional<services::staff::Accounts>& accounts) const {
        auto transformer{[](auto& account) {
            logic::SocialProfile profile;
            profile.profile = *account.value;
            profile.type = std::vector<std::string>{*account.type};
            return profile;
        }};

        return makeFromAccounts<logic::SocialProfile>(socialProfiles, accounts, std::move(transformer));
    }

    void addWorkPhone(std::vector<logic::TelephoneNumber>& result,
            std::optional<std::string> workPhone) const {
        if (workPhone && (!workPhone->empty())) {
            logic::TelephoneNumber phone;
            phone.telephone_number = std::move(workPhone);
            result.push_back(std::move(phone));
        }
    }

    void addPhones(std::vector<logic::TelephoneNumber>& result,
            std::optional<services::staff::Phones> phones) const {
        if (!phones) {
            return;
        }

        auto transformer{[](auto& phone) {
            logic::TelephoneNumber telephoneNumber;
            telephoneNumber.telephone_number = std::move(phone.number);
            if (phone.type && (!phone.type->empty())) {
                telephoneNumber.type = std::vector<std::string>{std::move(*phone.type)};
            }

            return telephoneNumber;
        }};

        boost::copy(*phones | boost::adaptors::transformed(std::move(transformer)),
                std::back_inserter(result));
    }

    std::optional<std::vector<logic::TelephoneNumber>> makeTelephoneNumbers(
            std::optional<std::string> workPhone, std::optional<services::staff::Phones> phones) const {
        std::optional<std::vector<logic::TelephoneNumber>> result;
        if ((workPhone && (!workPhone->empty())) || phones) {
            result = std::vector<logic::TelephoneNumber>{};
            addWorkPhone(*result, std::move(workPhone));
            addPhones(*result, std::move(phones));
        }

        return result;
    }

    std::optional<std::vector<std::string>> makeNotes(
            std::optional<services::staff::Personal>& personal) const {
        std::optional<std::vector<std::string>> result;
        if (personal && personal->about && (!personal->about->empty())) {
            result = std::vector<std::string>{std::move(*personal->about)};
        }

        return result;
    }

    std::optional<std::vector<logic::Event>> makeEvents(
            std::optional<services::staff::Personal>& personal) const {
        std::optional<std::vector<logic::Event>> result;
        if (personal && personal->birthday && (!personal->birthday->empty())) {
            result = {{makeBirthdayEvent(std::move(*personal->birthday))}};
        }

        return result;
    }

    std::optional<std::vector<logic::Organization>> makeOrganizations(
            std::optional<services::staff::Position> position) const {
        std::optional<std::vector<logic::Organization>> result;
        if (position && position->ru && (!position->ru->empty())) {
            logic::Organization organization;
            organization.title = std::move(position->ru);
            result = std::vector<logic::Organization>{std::move(organization)};
        }

        return result;
    }

    logic::Vcard makeVcard(services::staff::Person person) const {
        preprocessPerson(person);

        logic::Vcard vcard;
        vcard.names = makeNames(std::move(person.name));
        vcard.emails = makeEmails(std::move(person.work_email));
        vcard.instant_messengers = makeInstantMessengers(person.accounts);
        vcard.social_profiles = makeSocialProfiles(person.accounts);
        vcard.telephone_numbers = makeTelephoneNumbers(std::move(person.work_phone),
                std::move(person.phones));
        vcard.notes = makeNotes(person.personal);
        vcard.events = makeEvents(person.personal);
        vcard.organizations = makeOrganizations(std::move(person.official.position));
        return vcard;
    }

    logic::NewContact makeNewContact(services::staff::Person person) const {
        logic::NewContact contact;
        contact.vcard = makeVcard(std::move(person));
        return contact;
    }

    logic::UpdatedContact makeUpdatedContact(logic::ContactId contactId,
            services::staff::Person person) const {
        logic::UpdatedContact contact;
        contact.contact_id = contactId;
        contact.vcard = makeVcard(std::move(person));
        return contact;
    }

    void processNewPerson(services::staff::Person person, std::vector<logic::NewContact>& contactsToCreate,
            std::vector<common::KeyId>& keyIdsToAdd) const {
        if (!person.official.is_dismissed) {
            keyIdsToAdd.emplace_back(person.id);
            contactsToCreate.emplace_back(makeNewContact(std::move(person)));
        }
    }

    void processExistingPerson(
            services::staff::Person person,
            logic::ContactId contactId,
            std::vector<logic::ContactId>& contactIdsToDelete,
            std::vector<logic::UpdatedContact>& contactsToUpdate,
            std::vector<common::KeyId>& keyIdsToDelete) const {
        if (!person.official.is_dismissed) {
            contactsToUpdate.emplace_back(makeUpdatedContact(contactId, std::move(person)));
        } else {
            contactIdsToDelete.emplace_back(contactId);
            keyIdsToDelete.emplace_back(person.id);
        }
    }

    using Uids = std::vector<services::staff::Uid>;
    void processPerson(
            services::staff::Person person,
            Uids& uids,
            const common::ContactIdSearchMap& searchMap,
            common::UpdateContactsData& updateContactsData,
            common::UpdateIdMapData& updateIdMapData) const {
        if (!person.official.is_dismissed) {
            uids.emplace_back(person.uid);
        }

        const auto element{searchMap.find(person.id)};
        if (element == searchMap.end()) {
            processNewPerson(std::move(person), updateContactsData.contactsToCreate,
                    updateIdMapData.keyIdsToAdd);
        } else {
            processExistingPerson(std::move(person), element->second.contact_id, updateContactsData.contactIdsToDelete,
                    updateContactsData.contactsToUpdate, updateIdMapData.keyIdsToDelete);
        }
    }

    expected<Uids> addNewStaffDataImpl(
            std::vector<common::IdMapElement> idMapElements,
            common::ContactsInfo& contactsInfo,
            common::UpdateContactsData& updateContactsData,
            common::UpdateIdMapData& updateIdMapData) const {
        std::unordered_set<common::KeyId> receivedKeyIds;
        Uids uids;
        auto contactInfoIdMap{common::makeContactInfoIdMap(contactsInfo)};
        const auto searchMap{common::makeContactIdSearchMap(idMapElements, contactInfoIdMap)};
        auto page{0u};
        for (; page < config.services.staff.pageMaxCount; ++page) {
            if (workerContext->mustStop()) {
                return make_expected_from_error<Uids>(error_code{Error::syncInterrupted});
            }

            auto persons{staff->getPersons(workerContext->getTaskContext())};
            if (!persons) {
                return make_expected_from_error<Uids>(std::move(persons).error());
            }

            if (persons.value().empty()) {
                break;
            }

            const auto duplicate{[&](const auto& person){
                    return receivedKeyIds.find(person.id) != receivedKeyIds.end();}};
            const auto logDuplicate{[&](auto id){LOGDOG_(workerContext->logger(), warning,
                    log::message="duplicated ID received from Staff", log::staff_id=id);}};

            for (auto& person : persons.value()) {
                if (duplicate(person)) {
                    logDuplicate(person.id);
                } else {
                    receivedKeyIds.emplace(person.id);
                    processPerson(std::move(person), uids, searchMap, updateContactsData, updateIdMapData);
                }
            }
        }

        if (page == config.services.staff.pageMaxCount) {
            LOGDOG_(workerContext->logger(), error, log::message="Staff page count limit exceeded");
        }

        return uids;
    }

    template<typename ContactsProvider> expected<Uids> prepareDataImpl(
            ContactsProvider& contactsProvider,
            logic::ListId orgListId,
            std::vector<common::IdMapElement> idMapElements,
            common::UpdateContactsData& updateContactsData,
            common::UpdateIdMapData& updateIdMapData) const {
        auto getDbCorrectionData{[&](auto&& contactIds) {
            return common::getDbCorrectionData(std::move(contactIds), std::move(idMapElements),
                    updateContactsData.contactIdsToDelete, updateIdMapData.keyIdsToDelete);
        }};

        auto addNewData{[&](auto&& actualIdMapElements) {
            auto addNewStaffData{[&](auto&& contactsInfo){return addNewStaffDataImpl(std::move(
                actualIdMapElements), contactsInfo, updateContactsData, updateIdMapData);}};
            
            auto contactIds{common::getKeyIdsFromIdMapElements(idMapElements)};
            using services::db::contacts::query::GetContacts;
            return logic::db::contacts::getContacts<GetContacts>(contactsProvider, contactIds,
                    {orgListId}, {},{}).bind(std::move(addNewStaffData));
        }};

        return logic::db::contacts::getContactIdsByListIds(contactsProvider, {orgListId}, {}).bind(std::move(
                getDbCorrectionData)).bind(std::move(addNewData));
    }

    template<typename EventsQueueTransaction, typename ContactsProvider> expected<void> processDataImpl(
            EventsQueueTransaction& eventsQueueTransaction,
            ContactsProvider& contactsProvider,
            common::UpdateContactsData updateContactsData,
            common::UpdateIdMapData updateIdMapData,
            logic::ListId orgListId,
            Uids uids) const {
        auto transact{[&](auto&& contactsTransaction) {
            auto updateDb{[&]{return updateDbImpl(eventsQueueTransaction, contactsTransaction,
                    std::move(updateContactsData), std::move(updateIdMapData));}};
            auto returnTransaction{[&]{return std::move(contactsTransaction);}};
            return updateDb().bind(std::move(returnTransaction));
        }};

        auto commit{[&](auto&& contactsTransaction){return services::db::commit(contactsProvider,
                std::move(contactsTransaction));}};
        auto ignoreResult{[](auto&&){}};

        auto shareLists{[&]{return shareListsImpl(contactsProvider, orgListId, std::move(uids));}};

        return services::db::begin(contactsProvider).bind(std::move(transact)).bind(std::move(commit)).bind(
                std::move(ignoreResult)).bind(std::move(shareLists));
    }

    template<typename EventsQueueTransaction, typename ContactsTransaction> expected<void> updateDbImpl(
            EventsQueueTransaction& eventsQueueTransaction,
            ContactsTransaction& contactsTransaction,
            common::UpdateContactsData updateContactsData,
            common::UpdateIdMapData updateIdMapData) const {
        using AddStaffIdMapElements = services::db::events_queue::query::AddStaffIdMapElements;
        using DeleteContactIdsByStaffIds = services::db::events_queue::query::DeleteContactIdsByStaffIds;
        common::UpdateDb<EventsQueueTransaction, ContactsTransaction, AddStaffIdMapElements,
                DeleteContactIdsByStaffIds> updateDb{eventsQueueTransaction, contactsTransaction,
                        config.chunkSizeForModifyingOp};
        auto updateIdMap{[&](auto&& contactIds){return updateDb(std::move(updateIdMapData), std::move(
                contactIds));}};
        return updateDb(std::move(updateContactsData)).bind(std::move(updateIdMap));
    }

    template<typename ContactsProvider> expected<void> shareListsImpl(ContactsProvider& contactsProvider,
            logic::ListId orgListId, Uids actualUserIds) const {
        auto sharedLists{logic::db::contacts::getSharedLists(contactsProvider)};
        if (!sharedLists) {
            return make_expected_from_error<void>(std::move(sharedLists).error());
        }

        const auto filter{[&](const auto& list){return list.list_id == orgListId;}};
        const auto listTransformer{[](const auto& list){return list.client_user_id;}};
        Uids existingUserIds;
        boost::copy(*sharedLists | boost::adaptors::filtered(std::move(filter)) | boost::adaptors::
                transformed(std::move(listTransformer)), std::back_inserter(existingUserIds));
        if (!std::is_sorted(existingUserIds.begin(), existingUserIds.end())) {
            std::sort(existingUserIds.begin(), existingUserIds.end());
        }

        if (!std::is_sorted(actualUserIds.begin(), actualUserIds.end())) {
            std::sort(actualUserIds.begin(), actualUserIds.end());
        }

        actualUserIds.erase(std::unique(actualUserIds.begin(), actualUserIds.end()), actualUserIds.end());

        Uids userIdsToUnsubscribe;
        std::set_difference(existingUserIds.begin(), existingUserIds.end(), actualUserIds.begin(),
                actualUserIds.end(), std::back_inserter(userIdsToUnsubscribe));

        Uids userIdsToSubscribe;
        std::set_difference(actualUserIds.begin(), actualUserIds.end(), existingUserIds.begin(),
                existingUserIds.end(), std::back_inserter(userIdsToSubscribe));

        auto unsubscribe{[&](auto&& userProvider){return logic::db::contacts::unsubscribeFromContacts(
                contactsProvider, std::move(userProvider), orgListId);}};
        manageSubscriptions(std::move(userIdsToUnsubscribe), std::move(unsubscribe));

        const std::string clientListName{"staff_contacts"};
        auto subscribe{[&](auto&& userProvider){return logic::db::contacts::subscribeToContacts(
                contactsProvider, std::move(userProvider), orgListId, clientListName);}};
        manageSubscriptions(std::move(userIdsToSubscribe), std::move(subscribe));

        return make_expected();
    }

    template<typename Handler> void manageSubscriptions(Uids ids, Handler handler) const {
        for (auto id : ids) {
            if (workerContext->mustStop()) {
                LOGDOG_(workerContext->logger(), error, log::error_code=error_code{Error::syncInterrupted},
                        log::message="Staff subscription managing failed");
                return;
            }

            auto provider{makeContactsConnectionProvider(workerContext->getTaskContext(),
                    services::db::PassportUserId{id})};
            auto connection{getConnection(provider)};
            if (connection) {
                auto result{handler(std::move(provider))};
                if (!result) {
                    LOGDOG_(workerContext->logger(), error, log::error_code=std::move(result).error(),
                            log::user_id=id, log::message="Handler error while subscription managing");
                }
            } else {
                LOGDOG_(workerContext->logger(), warning, log::error_code=std::move(connection).error(),
                        log::user_id=id, log::message="connection getting error while subscription managing");
            }
        }
    }

    common::WorkerContextPtr workerContext;
    const MakeContactsConnectionProvider& makeContactsConnectionProvider;
    const MakeEventsQueueConnectionProvider& makeEventsQueueConnectionProvider;
    const StaffClientPtr& staff;
    const Config& config;
    const std::vector<std::string> instantMessengers;
    const std::vector<std::string> socialProfiles;
};

} // namespace collie::sync::staff
