#pragma once

#include <algorithm>
#include <mail/unistat/cpp/include/meters/common.h>
#include <re2/re2.h>
#include <boost/algorithm/string.hpp>
#include <spdlog/details/format.h>

namespace unistat {


namespace detail {


inline std::string getStatusCode(const std::string& code) {
    static const std::string strange = "0";
    if (code.size() != 3) {
        return strange;
    }

    if (code.front() < '1' || code.front() > '5') {
        return strange;
    }
    return code;
}

inline std::string getMethod(const std::string& method) {
    static const std::string strange = "strange_method";
    static const std::string METHODS[] = {"GET", "HEAD", "POST", "PUT", "DELETE", "CONNECT", "OPTIONS", "TRACE", "PATCH"};

    if (method.empty()) {
        return strange;
    }

    auto res = std::find_if(begin(METHODS), end(METHODS), [&method](const auto& s) {return boost::iequals(s, method);} );
    if (res == end(METHODS)) {
        return strange;
    }

    return *res;
}

}


struct CountByPath {

    explicit CountByPath(std::string reWithNamedCapGroups, std::string namePrefix, std::string nameSuffix)
            : _regex{reWithNamedCapGroups}
            , _prefix{normalizeName(std::move(namePrefix))}
            , _suffix{normalizeName(std::move(nameSuffix))}
            , _counters{} {
        
        if (!_regex.ok()) {
            throw std::invalid_argument(_regex.error());
        }

        if (_regex.NumberOfCapturingGroups() <= 0) {
            throw std::invalid_argument("there are no named capturing group");
        }

        const auto numCaptures = static_cast<size_t>(_regex.NumberOfCapturingGroups() + 1);
        _matches.resize(numCaptures);
        
        constexpr size_t degreesOfFreedom = 3;
        _counters.reserve(numCaptures * degreesOfFreedom);
    }

    void update(const std::map<std::string, std::string>& record) {
        const auto valueOrEmpty = [&record] (const auto& key) -> const std::string& {
            const auto it = record.find(key);
            if (it == record.cend()) {
                static const std::string s;
                return s;
            }
            return it->second;
        };

        const std::string& req = valueOrEmpty("request");
        if (req.empty()) {
            return;
        }

        for (auto& m : _matches) {
            m.set(nullptr);
        }
        const auto matchesSize = static_cast<int>(_matches.size());
        if (!_regex.Match(req.data(), 0, req.size(), re2::RE2::Anchor::ANCHOR_BOTH, _matches.data(), matchesSize)) {
            return;
        }

        const std::string method = detail::getMethod(valueOrEmpty("method"));
        const std::string code = detail::getStatusCode(valueOrEmpty("status_code"));
        const auto& groupNames = _regex.CapturingGroupNames();
        for (int idx = 1; idx < matchesSize; ++idx) {
            if (_matches[idx].empty()) {
                continue;
            }
            
            increment(groupNames.at(idx), method, code);
        }

    }

    std::vector<NamedValue<std::size_t>> get() const {
        std::vector<NamedValue<std::size_t>> result;
        result.reserve(_counters.size());
        std::transform(_counters.begin(), _counters.end(), std::back_inserter(result), [] (const auto& p) {
            return NamedValue<std::size_t>{p.first, p.second};
        });
        return result;
    }


private:
    const re2::RE2 _regex;
    const std::string _prefix;
    const std::string _suffix;

    std::unordered_map<std::string, std::size_t> _counters;

    std::vector<re2::StringPiece> _matches;

    void increment(const std::string& path, const std::string& method, const std::string& status) {
        std::string counter = fmt::format("{}_{}_{}_{}_{}", _prefix, path, method, status, _suffix);
        ++_counters[normalizeName(std::move(counter))];
    }
};


}

