from argparse import ArgumentParser
from collections import namedtuple
from mail.unistat.cpp.cython.meters import (
    AccessLogCount,
    AccessLogCountByFirstStatusDigit,
    AccessLogRequestTimeHist,
    HttpClientHttpRequestTotalTimeHist,
    HttpClientHttpRequestCountByStatus,
    SupervisorLogRestartMeters,
)
from mail.collie.unistat.cpp.run import (
    run as run_unistat,
    CountByPath,
    ServiceLog
)
import os
import logging
import mail.unistat.cpp.cython.logs as logs
import re
from unistat.utils import (
    read_yplatform_config,
)

logging.basicConfig(level=logging.WARNING, format='[%(asctime)s] [%(levelname)s]: %(message)s')
log = logging.getLogger(__name__)


HTTP_HIST_BUCKETS = (0, 20, 100, 300, 1000)


def make_access_log_meters(name_prefix):
    return [
        AccessLogCount(name_prefix),
        AccessLogCountByFirstStatusDigit(name_prefix),
        AccessLogRequestTimeHist(HTTP_HIST_BUCKETS, 'access_log_request'),
    ]


def make_count_by_path_meter():
    contact_id = '[0-9]{1,}'
    contact_ids = '\\[({contactid}(?:,\\s*{contactid})*)*]'.format(contactid=contact_id)
    pattern = r'''
    ^(?:
        (?:/v1/users/{uid}/)
        (?:
            (?P<contacts_emails_signal>contacts/emails)|
            (?P<contacts_count_signal>contacts/count)|
            (?P<contacts_signal>contacts)|
            (?P<contacts_contact_ids_signal>contacts/{contactids})|
            (?P<update_contacts_signal>update_contacts)|
            (?P<emails_tag_ids_signal>emails/\[{tagid}(?:,\s*{tagid})*])|
            (?P<tags_signal>tags)|
            (?P<tags_tag_id_signal>tags/{tagid})|
            (?P<tags_tag_id_contacts_signal>tags/{tagid}/contacts)|
            (?P<changes_signal>changes)|
            (?P<changes_revision_signal>changes/{revision})|
            (?P<location_signal>location)|
            (?P<carddav_multiget_signal>carddav/multiget)|
            (?P<carddav_propfind_signal>carddav/propfind)|
            (?P<carddav_uri_delete_signal>carddav/{uri}/delete)|
            (?P<carddav_uri_put_etag_signal>carddav/{uri}/put/{etag})|
            (?P<shared_lists_signal>shared/lists)|
            (?P<shared_lists_list_id_contacts_count_signal>shared/lists/{listid}/contacts/count)|
            (?P<shared_lists_list_id_contacts_contact_ids_signal>shared/lists/{listid}/contacts/{contactids})
        )|
        (?:/v1/)
        (?:
            (?P<organizations_events_signal>organizations/events)|
            (?P<search_contacts_signal>searchContacts)
        )|
        (?:/compat/)
        (?:
            (?P<colabook_feed_addrdb_signal>colabook_feed_addrdb)
        )
    )
    (?:\s+|[?]|$)
    '''.format(
        uid='[0-9]{1,}',
        contactids=contact_ids,
        tagid='[0-9]{1,}',
        revision='[0-9]{1,}',
        uri='.+?',
        etag='.+?',
        listid='[0-9]{1,}'
    )
    pattern = re.sub(r'\s+', '', pattern)
    return [CountByPath(pattern, 'collie', 'summ')]


def make_supervisor_meters():
    return [
        SupervisorLogRestartMeters('supervisor'),
    ]


def parse_args():
    parser = ArgumentParser()
    parser.add_argument('-H', '--host', default='::')
    parser.add_argument('-p', '--port', default=8082, type=int)
    parser.add_argument('-d', '--dir', default='.')
    parser.add_argument('-s', action='store_true', help='read file logs from start')
    parser.add_argument('-l', '--log', default='/var/log/collie/unistat.log', help='path for yplatform.log')
    parser.add_argument('--supervisorlog', help='path for supervisord.log')
    parser.add_argument('service_config')
    return parser.parse_args()


def get_module_config(data, name):
    res = next((v for v in data['config']['modules']['module'] if v['system']['name'] == name), None)
    return res['configuration'] if res else None


def get_logger_sink_path(data, name, sink_index=0):
    return os.path.join(os.curdir, data['config']['log'][name]['sinks'][sink_index]['path'])


def make_service_config(data):
    remove_scheme = re.compile(r"https?://")

    service_name = 'collie'
    service_config = get_module_config(data, service_name)
    if not service_config:
        service_config = get_module_config(data, 'collie_staff_sync')

    services = service_config['services']
    events_queue_sharpei_address = services['db']['events_queue']['sharpei']['client']['address']
    org_sharpei_address = services['db']['contacts']['org_sharpei']['client']['address']
    user_sharpei_address = services['db']['contacts']['user_sharpei']['client']['address']

    return ServiceConfig(
        httpclient_log=get_logger_sink_path(data, 'client'),
        access_log=get_logger_sink_path(data, 'access'),

        events_queue_sharpei_host='{host}:{port}'.format(
            host=remove_scheme.sub('', events_queue_sharpei_address['host']).strip(),
            port=events_queue_sharpei_address['port']
        ),
        org_sharpei_host='{host}:{port}'.format(host=remove_scheme.sub('', org_sharpei_address['host']).strip(), port=org_sharpei_address['port']),
        user_sharpei_host='{host}:{port}'.format(host=remove_scheme.sub('', user_sharpei_address['host']).strip(), port=user_sharpei_address['port']),
        directory_host=remove_scheme.sub('', services['directory']['location']).strip(),
        ml_host=remove_scheme.sub('', services['ml']['location']).strip(),
        staff_host=remove_scheme.sub('', services['staff']['location']).strip(),
    )


def make_http_client_log_meters(cfg):
    meters = []
    host2name = [
        (cfg.events_queue_sharpei_host,   'events_queue_sharpei'),
        (cfg.user_sharpei_host,           'user_sharpei'),
        (cfg.org_sharpei_host,            'org_sharpei'),
        (cfg.directory_host,              'directory'),
        (cfg.staff_host,                  'staff'),
        (cfg.ml_host,                     'ml'),
    ]

    for host, name in host2name:
        if not host:
            continue

        meters += [
            HttpClientHttpRequestCountByStatus(host, 'count_by_status_{}'.format(name)),
            HttpClientHttpRequestTotalTimeHist(HTTP_HIST_BUCKETS, host, 'time_hist_{}'.format(name)),
        ]

    return meters


ServiceConfig = namedtuple('ServiceConfig', (
    'httpclient_log',
    'access_log',

    'events_queue_sharpei_host',
    'org_sharpei_host',
    'user_sharpei_host',
    'directory_host',
    'ml_host',
    'staff_host'
))


def main():
    args = parse_args()
    log.info('chdir %s' % os.path.abspath(args.dir))
    os.chdir(args.dir)

    config = read_yplatform_config(args.service_config)
    config = make_service_config(config)

    fast_forward = args.s

    access_log = logs.AccessTskv([], make_access_log_meters('access_log'), fast_forward, config.access_log)
    count_by_path = ServiceLog([], make_count_by_path_meter(), fast_forward, config.access_log)

    http_client_log = logs.HttpClientLog([], make_http_client_log_meters(config), fast_forward, config.httpclient_log)

    logs_list = [access_log, count_by_path, http_client_log]

    if args.supervisorlog is not None:
        supervisor_log = logs.SupervisorLog(
            [],
            make_supervisor_meters(),
            fast_forward,
            args.supervisorlog
        )
        logs_list.append(supervisor_log)

    run_unistat(args.host, args.port, logs_list, args.log, logLevel='info')


if __name__ == '__main__':
    main()
