#!/usr/bin/env python2

import sys
import requests

from collections import namedtuple
from itertools import takewhile


def main():
    old_lines = list(read_lines(sys.stdin))
    new_lines = list(read_lines(get_new()))
    values = {v.mime_type: v for v in old_lines if isinstance(v, (OnlyMimeType, WithFormats))}
    for new_line in new_lines:
        if isinstance(new_line, Header):
            continue
        old_value = values.get(new_line.mime_type)
        if old_value is None or isinstance(new_line, OnlyMimeType):
            values[new_line.mime_type] = new_line
        elif isinstance(old_value, WithFormats) and isinstance(new_line, WithFormats):
            values[new_line.mime_type] = WithFormats(
                mime_type=old_value.mime_type,
                separator=old_value.separator,
                formats=merge(old_value.formats, new_line.formats),
            )
    for line in takewhile(lambda v: isinstance(v, Header), new_lines):
        sys.stdout.write('%s\n' % line.content)
    for line in sorted(values.values(), key=lambda v: v.mime_type):
        if isinstance(line, OnlyMimeType):
            sys.stdout.write('# %s\n' % line.mime_type)
        elif isinstance(line, WithFormats):
            sys.stdout.write('%s%s%s\n' % (line.mime_type, line.separator, ' '.join(line.formats)))


def get_new():
    response = requests.get(URL)
    assert response.status_code == 200
    return response.iter_lines()


def read_lines(stream):
    header = True
    for line in stream:
        if header:
            if line.lstrip('# ').startswith('='):
                header = False
            yield Header(content=line)
            continue
        if line.lstrip().startswith('#'):
            mime_type = line[line.find('#') + 1:].strip()
            assert '\t' not in mime_type
            yield OnlyMimeType(mime_type=mime_type)
        else:
            parts = line.strip().split('\t')
            assert len(parts) >= 2
            yield WithFormats(
                mime_type=parts[0],
                separator='\t' * (len(parts[1:-1]) + 1),
                formats=parts[-1].split(' '),
            )


def merge(lhs, rhs):
    result = list(lhs)
    values = set(lhs)
    for v in rhs:
        if v not in values:
            values.add(v)
            result.append(v)
    return result


URL = 'http://svn.apache.org/repos/asf/httpd/httpd/trunk/docs/conf/mime.types'


Header = namedtuple('Header', ('content',))
OnlyMimeType = namedtuple('OnlyMimeType', ('mime_type',))
WithFormats = namedtuple('WithFormats', ('mime_type', 'separator', 'formats'))


if __name__ == '__main__':
    main()
