#pragma once

#include <boost/asio/spawn.hpp>
#include <boost/asio/post.hpp>

#include <atomic>
#include <queue>
#include <mutex>
#include <sstream>

namespace coro {

/**
 * Mutex implementation for boost::asio::yeld_context based coroutines.
 * The general difference between this class and standard mutex is that
 * all the operations should use LockHandle - handle for current lock.
 */
class Mutex {
public:
    using YieldContext = boost::asio::yield_context;
    using LockHandle = const void*;

    /**
     * Locks mutex or suspend the coroutine with given context until lock. Initially
     * the function tries to switch between context specified number of spins
     * like spinlock. If lock is not succeeded then the given context being
     * stored in the internal queue and the coroutine will be suspended.
     *
     * @param yield --- coroutine yield context.
     * @param spins --- number of context switch spins to
     * @return LockHandle --- handle of current lock
     */
    LockHandle lock(const YieldContext& yield, int spins = 1000) {
        for (;spins > 0; --spins) {
            if (try_own(&yield)) {
                return &yield;
            }
            yield_coro(yield);
        }

        std::unique_lock<std::mutex> guard(lock_);
        if (!try_own(&yield)) {
            enqueue(yield);
            guard.unlock();
            suspend(yield);
            set_owner(&yield);
        }

        return &yield;
    }

    /**
     * Try to lock the mutext by coroutine with given context.
     *
     * @param yield --- coroutine context
     * @return LockHandle --- handle of the current lock, or nullptr if lock failed
     */
    LockHandle try_lock(const YieldContext& yield) {
        return try_own(&yield) ? &yield : nullptr;
    }

    /**
     * Unlock mutex that is locked by the coroutine. Note that there is no context
     * switch in this operation. The next coroutine awakeness is scheduled via
     * posting to the associated executor.
     *
     * @param handle --- LockHandle is obtained with lock() or try_lock() funtions.
     */
    void unlock(LockHandle handle) {
        const auto owner = get_owner();
        if (owner != handle) {
            std::ostringstream stream;
            stream << "Mutex::unlock: trying to unlock by not lock owner, where owner=" << owner << ", handle=" << handle;
            throw std::logic_error(stream.str());
        }

        std::unique_lock<std::mutex> guard(lock_);
        if (has_next()) {
            auto next = get_next();
            guard.unlock();
            awake(std::move(next));
        } else {
            set_owner(nullptr);
        }
    }

private:
    struct Consumer {
        boost::asio::detail::shared_ptr<YieldContext::callee_type> coro;
        boost::asio::executor executor;
    };

    static_assert(std::is_nothrow_move_constructible<Consumer>::value, "Consumer must be nothrow move constructible");

    void set_owner(LockHandle h) {
        owner_ = h;
    }

    LockHandle get_owner() const {
        return owner_;
    }

    bool try_own(LockHandle h) {
        LockHandle null = nullptr;
        return owner_.compare_exchange_strong(null, h);
    }

    void enqueue(YieldContext yield) {
        queue_.push(Consumer {yield.coro_.lock(), boost::asio::get_associated_executor(yield.handler_)});
    }

    bool has_next() const {
        return !queue_.empty();
    }

    Consumer get_next() {
        auto next = std::move(queue_.front());
        queue_.pop();
        return next;
    }

    static void suspend(YieldContext yield) {
        yield.ca_();
    }

    static void awake(Consumer consumer) {
        boost::asio::post(consumer.executor, [coro = std::move(consumer.coro)] { (*coro)(); });
    }

    static void yield_coro(YieldContext yield) {
        boost::asio::post(yield);
    }

    std::mutex lock_;
    std::atomic<LockHandle> owner_{nullptr};
    std::queue<Consumer> queue_;
};

namespace detail {

using mutex_type = Mutex;

template <typename Mutex>
using handle_type = typename Mutex::LockHandle;

template <typename Mutex>
using yield_context = typename Mutex::YieldContext;

template <typename Mutex>
using lock_data = std::tuple<Mutex*, handle_type<Mutex>>;

template <typename Mutex>
inline auto handle(const lock_data<Mutex>& v) {
    return std::get<handle_type<Mutex>>(v);
}

template <typename Mutex>
inline auto set_handle(lock_data<Mutex>& v, handle_type<Mutex> h) {
    return std::get<handle_type<Mutex>>(v) = h;
}

template <typename Mutex>
inline auto mutex(const lock_data<Mutex>& v) {
    return std::get<Mutex*>(v);
}

} // namespace detail

/**
 * Analog of std::lock_guard.
 */
class lock_guard {
public:
    using mutex_type = Mutex;
    using handle_type = detail::handle_type<Mutex>;
    using yield_context = detail::yield_context<Mutex>;

    lock_guard(mutex_type& m, const yield_context& yield)
     : lock_guard(m, m.lock(yield)) {}

    lock_guard(mutex_type& m, const yield_context& yield, int spins)
     : lock_guard(m, m.lock(yield, spins)) {}

    lock_guard(mutex_type& m, handle_type h) noexcept
     : v_{&m, h} {}

    ~lock_guard() {
        detail::mutex(v_)->unlock(detail::handle(v_));
    }
private:
    lock_guard(const lock_guard& ) = delete;

    detail::lock_data<Mutex> v_ = {nullptr, nullptr};
};

/**
 * Analog of std::unique_lock.
 */
template<typename Mutex>
class unique_lock {
public:
    using mutex_type = Mutex;
    using handle_type = detail::handle_type<Mutex>;
    using yield_context = detail::yield_context<Mutex>;

    unique_lock() noexcept {}

    unique_lock(unique_lock&& other) noexcept
     : v_(other.release()) {}

    unique_lock(mutex_type& m, const yield_context& yield)
     : unique_lock(m, m.lock(yield)) {}

    unique_lock(mutex_type& m, const yield_context& yield, int spins)
     : unique_lock(m, m.lock(yield, spins)) {}

    unique_lock(mutex_type& m, std::defer_lock_t) noexcept
     : unique_lock(m, nullptr) {}

    unique_lock(mutex_type& m, const yield_context& yield, std::try_to_lock_t)
     : unique_lock(m, m.try_lock(yield)) {}

    unique_lock(mutex_type& m, handle_type h) noexcept
     : v_{&m, h} {}

    ~unique_lock() {
        if(owns_lock()) {
            unlock();
        }
    }

    unique_lock& operator = (unique_lock&& rhs) {
        unlock();
        v_ = rhs.release();
        return *this;
    }

    bool owns_lock() const noexcept { return detail::handle(v_) != nullptr; }

    void lock(const yield_context& yield) {
        if(owns_lock()) {
            throw std::logic_error("unique_lock::lock(): mutex already locked");
        }
        detail::set_handle(v_, mutex_ref().lock(yield));
    }

    void lock(const yield_context& yield, int spins) {
        if(owns_lock()) {
            throw std::logic_error("unique_lock::lock(): mutex already locked");
        }
        detail::set_handle(v_, mutex_ref().lock(yield, spins));
    }

    bool try_lock(const yield_context& yield) {
        if(owns_lock()) {
            throw std::logic_error("unique_lock::try_lock(): mutex already locked");
        }
        detail::set_handle(v_, mutex_ref().try_lock(yield));
        return owns_lock();
    }

    void unlock() {
        if(!owns_lock()) {
            throw std::logic_error("unique_lock: mutex is not locked");
        }
        mutex_ref().unlock(detail::handle(v_));
        detail::set_handle(v_, nullptr);
    }

    std::tuple<mutex_type*, handle_type> release() noexcept {
        auto retval = v_;
        v_ = detail::lock_data<Mutex>{nullptr, nullptr};
        return retval;
    }

    void swap(unique_lock& other) {
        std::swap(v_, other.v_);
    }

    mutex_type* mutex() const noexcept { return detail::mutex(v_); }

    operator bool() const { return owns_lock(); }

private:
    detail::lock_data<Mutex> v_ = {nullptr, nullptr};

    unique_lock(const unique_lock&) = delete;
    unique_lock& operator = (const unique_lock&) = delete;

    mutex_type& mutex_ref() const {
        if(!mutex()) {
            throw std::logic_error("unique_lock: no mutex is associated");
        }
        return *mutex();
    }
};

} // namespace coro
