# coding=utf-8
import math
import os
import networkx as nx
import matplotlib.pyplot as plt
import yt.wrapper as yt
from networkx.drawing.nx_pydot import write_dot
from networkx.drawing.nx_pydot import graphviz_layout
from networkx.drawing.nx_pydot import pydot_layout

import utils


def module_filter(module):
    module_name = getattr(module, '__name__', '')
    if 'numpy' in module_name:
        return False
    if 'yt_yson_bindings' in module_name:
        return False
    if 'hashlib' in module_name:
        return False
    if 'hmac' in module_name:
        return False

    module_file = getattr(module, '__file__', '')
    if not module_file:
        return False
    if module_file.endswith('.so'):
        return False

    return True


yt.config["proxy"]["url"] = "hahn.yt.yandex.net"
yt.config["memory_limit"] = 6000000000
yt.config['pickling']['dynamic_libraries']['enable_auto_collection'] = True
yt.config['pickling']['module_filter'] = module_filter
yt.config["tabular_data_format"] = yt.YsonFormat()

yt.config["pool"] = 'sherlock'
yt.config["write_parallel"]["enable"] = True


def get_last_caused_by(ex_str):
    error_lines = ex_str.replace('\t', '').split('\n')
    start_line_num = 0
    for error_line in error_lines:
        if error_line.startswith('Caused by:'):
            start_line_num = error_lines.index(error_line)

    last_calls_stack = error_lines[start_line_num:]
    new_last_calls_stack = list()
    for line in last_calls_stack:
        new_line = line.strip().replace('Caused by: ', '')
        source_file_open_brace_index = -1
        source_file_close_brace_index = -1
        try:
            source_file_open_brace_index = new_line.index('(')
            source_file_close_brace_index = new_line.index(')')
        except:
            pass

        if source_file_open_brace_index != -1 and source_file_close_brace_index != -1 and 'at ' in new_line:
            source_file_with_line_num = new_line[source_file_open_brace_index:source_file_close_brace_index + 1]
            source_file_and_line_num_splitted = source_file_with_line_num.replace('(', '').replace(')', '').split(':')
            line_num = None
            if len(source_file_and_line_num_splitted) > 1:
                line_num = source_file_and_line_num_splitted[1]
            new_line = new_line.replace(source_file_with_line_num, '')
            if line_num and int(line_num) > 0:
                new_line = new_line + '({})'.format(line_num)
        if 'at ' in new_line:
            new_line = new_line.replace('at ', '')
        if '...' in new_line and ' more' in new_line:
            continue
        if 'Lambda$' in new_line or 'lambda$' in new_line:
            continue
        if '.' in new_line:
            new_last_calls_stack.append(new_line)

    if len(new_last_calls_stack) > 0 and ':' in new_last_calls_stack[0]:
        new_last_calls_stack[0] = new_last_calls_stack[0].split(':')[0]
        # remove message from first line
    return '\n'.join(new_last_calls_stack)

def split(delimiters, string, maxsplit=0):
    import re
    regexPattern = '|'.join(map(re.escape, delimiters))
    return re.split(regexPattern, string, maxsplit)


def visualize_graph(gr: nx.MultiDiGraph(), edges1, edges_labels1):
    write_dot(gr, 'test.dot')
    pos = pydot_layout(gr, prog='dot')
    plt.subplots(figsize=(150, 100))
    nx.draw_networkx_edge_labels(gr, pos, edge_labels=edges_labels1, font_size=20)
    nx.draw(gr, pos, edges=edges1, with_labels=True, node_size=20, width=10, font_size=20)
    plt.savefig("Graph4.png", format="PNG")
    # plt.show()


def prepare_graph_info(gr: nx.MultiDiGraph(), threshold: int):
    new_graph1 = nx.DiGraph()
    graph_edges1 = list()
    edges1 = gr.edges(data=True)
    weights1 = list()
    edges_labels1 = dict()
    weight_sum = 0
    whitelisted_packages = ['com', 'com.yandex']
    for edge in edges1:
        edge_weight = len(gr[edge[0]][edge[1]])
        if edge_weight > threshold:
            should_skip = False
            for excluded_package in whitelisted_packages:
                if not(excluded_package in edge[0] or excluded_package in edge[1]):
                    should_skip = True
            if should_skip:
                continue
            weights1.append(edge_weight)
            weight_sum += edge_weight
            label1 = edge[0]
            label2 = edge[1]
            edges_labels1[(label1, label2)] = str(edge_weight)
            graph_edges1.append((label1, label2, {'weight': float(edge_weight)}))

    for i in range(0, len(weights1) - 1):
        weights1[i] = math.exp(-weights1[i] / weight_sum)

    new_graph1.add_weighted_edges_from(graph_edges1)
    return new_graph1, edges1, edges_labels1, weights1

def build_graph_for_call_stack(gr: nx.MultiDiGraph, callstack_str: str):
    splitted = callstack_str
    if len(splitted) > 1:
        splitted = splitted[1:]
        for single_frame in splitted:
            single_frame_no_line_num = single_frame
            frames_decomposed = split(['.', '$'], single_frame_no_line_num)
            if len(frames_decomposed) > 0:
                frames_decomposed = frames_decomposed[:-1]

            # fqdn для фреймов
            for frame_index in range(1, len(frames_decomposed)):
                frames_decomposed[frame_index] = '{}.{}'.format(
                    frames_decomposed[frame_index - 1],
                    frames_decomposed[frame_index]
                )

            graph_edges = list()
            for frame_index in range(1, len(frames_decomposed)):
                frame = frames_decomposed[frame_index]
                next_frame = frames_decomposed[frame_index - 1]
                graph_edges.append((next_frame, frame, 1))
            gr.add_weighted_edges_from(graph_edges)


if __name__ == '__main__':
    graph = nx.MultiDiGraph()
    all_crashes = yt.read_table(
        '//home/mobilesearch/salavat/retrace/api_key_14836/version_4.17.2/deobfuscated/1d/2019-03-19',
        format=yt.JsonFormat())

    for crash in all_crashes:
        processed_tracktrace = utils.get_last_caused_by(crash['Trace'])
        # print('processed file {}'.format(crash))
        build_graph_for_call_stack(graph, processed_tracktrace)
    print('building edges, labels and weights...')
    new_graph, edges, edges_labels, weights = prepare_graph_info(graph, 100)
    print('visualizing graph...')
    visualize_graph(new_graph, edges, edges_labels)



