from __future__ import print_function

import argparse
import os
import pprint
import sys

import colorlog
from library.python.find_root import detect_root
from mail.devpack.lib.env import DevEnv
from mail.devpack.lib import config_master, components_manager
from mail.devpack.lib.coordinator import Coordinator
from IPython.core import ultratb


def find_arcadia_root():
    return detect_root(os.getcwd())


def init_logging():
    handler = colorlog.StreamHandler()
    handler.setFormatter(colorlog.ColoredFormatter(
        '%(log_color)s%(levelname)s:%(name)s:%(message)s')
    )
    logger = colorlog.getLogger()
    logger.addHandler(handler)


def wrap_main(deps_root):
    sys.excepthook = ultratb.FormattedTB(mode='Verbose', color_scheme='Linux', call_pdb=1)
    main(deps_root=deps_root)


def main(deps_root):
    return main_(sys.argv[1:], deps_root)


def main_(argv, deps_root):
    """Easier to run via interactive console, since you can't pass console arguments to script when running via repl"""
    init_logging()
    components = components_manager.names(deps_root)

    def add_component_arg(parser):
        parser.add_argument(
            'component',
            help='component to run command for',
            nargs='?',
            choices=components,
            default=deps_root.NAME
        )

    def add_deps_args(parser):
        parser.add_argument(
            '-w', '--without-deps',
            help='apply action only to component, without its dependencies',
            action='store_true'
        )
        parser.add_argument(
            '-o', '--only-deps',
            help='apply action only to dependencies, not to the component itself',
            action='store_true'
        )

    parser = argparse.ArgumentParser(
        description="Control script for Mail developer's pack",
        formatter_class=argparse.ArgumentDefaultsHelpFormatter
    )
    parser.add_argument(
        '-c', '--config',
        help='config for devpack_ctl',
        default=config_master.DEFAULT_CONFIG_PATH
    )

    subparsers = parser.add_subparsers(title='subcommands', dest='command')

    parser_start = subparsers.add_parser(
        'start',
        help='start component',
        description='Initialize and start component with prepared data. By default start component and its dependencies',
        formatter_class=argparse.ArgumentDefaultsHelpFormatter
    )
    add_component_arg(parser_start)
    add_deps_args(parser_start)
    parser_start.set_defaults(func=start)

    parser_stop = subparsers.add_parser(
        'stop',
        help='stop component',
        description='Stop component. By default stop component and its dependencies',
        formatter_class=argparse.ArgumentDefaultsHelpFormatter
    )
    add_component_arg(parser_stop)
    add_deps_args(parser_stop)
    parser_stop.set_defaults(func=stop)

    parser_restart = subparsers.add_parser(
        'restart',
        help='restart component',
        description='Restart component. By default restart component and its dependencies',
        formatter_class=argparse.ArgumentDefaultsHelpFormatter
    )
    add_component_arg(parser_restart)
    add_deps_args(parser_restart)
    parser_restart.set_defaults(func=restart)

    parser_purge = subparsers.add_parser(
        'purge',
        help='stop component and clear data',
        description='Stop component and clear its data in sysdata dir. By default purge component with its dependencies',
        formatter_class=argparse.ArgumentDefaultsHelpFormatter
    )
    add_component_arg(parser_purge)
    add_deps_args(parser_purge)
    parser_purge.add_argument(
        '--hard',
        help='find all processes by cwd and kill them after regular purge operations',
        action='store_true'
    )
    parser_purge.set_defaults(func=purge)

    parser_info = subparsers.add_parser(
        'info',
        help='show info about component',
        description='Show info about component. By default show info about component and its dependencies',
        formatter_class=argparse.ArgumentDefaultsHelpFormatter
    )
    add_component_arg(parser_info)
    add_deps_args(parser_info)
    parser_info.set_defaults(func=info)

    parser_interact = subparsers.add_parser(
        'interact',
        help='interact console',
        description='Dive into ipdb with loaded component',
        formatter_class=argparse.ArgumentDefaultsHelpFormatter
    )
    add_component_arg(parser_interact)
    parser_interact.set_defaults(func=interact)

    parser_communicate = subparsers.add_parser(
        'communicate',
        help='communicate with component',
        description='Dive into component-specific interactive communication',
        formatter_class=argparse.ArgumentDefaultsHelpFormatter
    )
    add_component_arg(parser_communicate)
    parser_communicate.set_defaults(func=communicate)

    parser_gencfg = subparsers.add_parser(
        'gen_config',
        help='generate default config and write it to default path',
        description='Generate default config and write it to default path',
        formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser_gencfg.add_argument(
        '-a', '--arcadia-root',
        help='specify your arcadia root',
        default=find_arcadia_root()
    )
    parser_gencfg.add_argument(
        '-o', '--offset',
        help='offset for services ports',
        default=6000,
        type=int
    )
    parser_gencfg.set_defaults(func=gencfg)

    args = parser.parse_args(argv)
    args.func(args, deps_root)
    return 0


def start(args, deps_root):
    component_cls = components_manager.find(args.component, deps_root)
    coordinator = Coordinator(DevEnv(config_master.read(args.config)), top_comp_cls=component_cls)
    if args.without_deps:
        coordinator.start(component_cls, with_deps=False)
    elif args.only_deps:
        coordinator.apply_action_for_deps(coordinator.start, component_cls)
    else:
        coordinator.start(component_cls)


def stop(args, deps_root):
    component_cls = components_manager.find(args.component, deps_root)
    coordinator = Coordinator(DevEnv(config_master.read(args.config)), top_comp_cls=component_cls)
    if args.without_deps:
        coordinator.stop(component_cls, with_deps=False)
    elif args.only_deps:
        coordinator.apply_action_for_deps(coordinator.stop, component_cls)
    else:
        coordinator.stop(component_cls)


def restart(args, deps_root):
    stop(args, deps_root)
    start(args, deps_root)


def purge(args, deps_root):
    component_cls = components_manager.find(args.component, deps_root)
    coordinator = Coordinator(DevEnv(config_master.read(args.config)), top_comp_cls=component_cls)
    if args.without_deps:
        coordinator.purge(component_cls, with_deps=False)
    elif args.only_deps:
        coordinator.apply_action_for_deps(coordinator.purge, component_cls)
    else:
        coordinator.purge(component_cls)
    if args.hard:
        coordinator.hard_purge()


def info(args, deps_root):
    config = config_master.read(args.config)
    component_cls = components_manager.find(args.component, deps_root)
    coordinator = Coordinator(DevEnv(config), top_comp_cls=component_cls)
    res = {"config_path": args.config}
    if args.without_deps:
        res.update(coordinator.info(component_cls, with_deps=False))
    elif args.only_deps:
        coordinator.apply_action_for_deps(coordinator.info, component_cls)
    else:
        res.update(coordinator.info(component_cls))
    pprint.pprint(res)


def gencfg(args, deps_root):
    config = config_master.generate_default_config(args.arcadia_root, args.offset, deps_root)
    config_master.write(config, args.config)
    print("config was written to " + args.config)


def interact(args, deps_root):
    config = config_master.read(args.config)
    component_cls = components_manager.find(args.component, deps_root)
    cr = coordinator = Coordinator(DevEnv(config), top_comp_cls=component_cls)  # noqa
    comp = coordinator.components[component_cls]

    import IPython
    IPython.embed(
        header='''Welcome to Devpack Interactive Mode! To begin, take:
         * `cr = coordinator: mail.devpack.lib.Coordinator` object holding all your dependency graph
         * `comp: {component_cls.__module__}.{component_cls.__name__}` for your top-level component'''.format(**locals())
    )


def communicate(args, deps_root):
    config = config_master.read(args.config)
    component_cls = components_manager.find(args.component, deps_root)
    coordinator = Coordinator(DevEnv(config), top_comp_cls=component_cls)
    coordinator.components[component_cls].communicate()
