import io
import os
import subprocess
import tarfile
import time
from clickhouse_driver import Client
from mail.devpack.lib.db import if_server_not_start, if_server_start
from string import Template

from library.python import resource

from . import helpers


CH_SERVER_LOG = 'clickhouse.log'
CH_WRAPPER_LOG = 'ch-wrapper.log'


class ClickHouse(object):

    def __init__(self, dbname, http_port, tcp_port=9000, working_dir=None):
        self.dbname = dbname
        self.http_port = http_port
        self.tcp_port = tcp_port

        self.server_name = 'clickhouse_%s_%d' % (dbname, http_port)
        self.root = helpers.create_root(self.server_name, working_dir, use_ram_disk_if_available=True)
        self.logger = helpers.create_logger(__name__, self, CH_WRAPPER_LOG)

        self.pid_file = os.path.join(self.root, 'clickhouse.pid')
        self.log_file = os.path.join(self.root, CH_SERVER_LOG)
        self.config_file = os.path.join(self.root, 'etc', 'clickhouse-server', 'config.xml')

        self.logger.info('clickhouse root is %s', self.root)

    def _get_bin_dir(self):
        return os.path.join(self.root, 'usr', 'bin')

    def _get_bin(self, bin_file='clickhouse'):
        return os.path.join(self._get_bin_dir(), bin_file)

    def _update_config(self):
        for file in ('config', 'users'):
            config = resource.find('ch/%s.xml' % file).decode('utf-8')
            config = Template(config).safe_substitute(
                {
                    'DEVPACK_ROOT': self.root,
                    'DEVPACK_http_port': self.http_port,
                    'DEVPACK_tcp_port': self.tcp_port,
                }
            )
            config_file = os.path.join(self.root, 'etc', 'clickhouse-server', '%s.xml' % file)
            helpers.write2file(config, config_file)

    def extract_tar(self):
        resource_name = 'clickhouse.tgz'
        self.logger.info('start extracting %s to %s', resource_name, self.root)
        res = resource.find(resource_name)
        if not res:
            raise RuntimeError('No clickhouse resource found: {}'.format(resource_name))

        with io.BytesIO(res) as tar_obj:
            tar = tarfile.open(fileobj=tar_obj)
            tar.extractall(path=self.root)

        self._update_config()

        self.logger.info('extracting done')

    def _make_env(self):
        env = os.environ.copy()
        env['LANG'] = 'en_US.UTF-8'
        env['LC_MESSAGES'] = 'en_US.UTF-8'
        return env

    def _check_run_server(self):
        return os.path.exists(self.pid_file)

    @if_server_not_start
    def start(self):
        self.logger.info('starting clickhouse server on port %d', self.http_port)
        cmd = [
            self._get_bin(),
            'server',
            '--config-file=%s' % self.config_file,
            '--log-file=%s' % self.log_file,
            '--pidfile=%s' % self.pid_file,
            '--daemon',
        ]
        helpers.run_subprocess('clickhouse server start', self.logger, cmd, self._make_env(), self.root)
        time.sleep(1)
        assert self._check_run_server(), 'could not start clickhouse'

    @if_server_start
    def stop(self):
        self.logger.info('stopping clickhouse server on port %d', self.http_port)
        helpers.kill_proc_by_pid_path(self.pid_file)
        self.logger.info('clickhouse server stopped')

    @if_server_start
    def dropdb(self):
        self.logger.info('starting dropdb %s', self.dbname)

        self.execute('DROP DATABASE IF EXISTS %s' % self.dbname)

        self.logger.info('db %s dropped', self.dbname)

    @if_server_start
    def createdb(self):
        self.logger.info('starting createdb %s', self.dbname)

        self.execute('CREATE DATABASE IF NOT EXISTS %s' % self.dbname)

        self.logger.info('db %s created', self.dbname)

    def _do_exec(self, query, params=None):
        self.logger.info('will execute %s', query)
        try:
            client = Client('localhost', port=self.tcp_port, secure=False)
            res = client.execute(query, params)
            client.disconnect()
            return res
        except:
            self.logger.error('cannot query=%s', query)
            raise

    def query(self, query, **kwargs):
        return self._do_exec(query, params=kwargs)

    def execute(self, query, **kwargs):
        return self._do_exec(query, params=kwargs)

    def info(self):
        return {
            'dbname': self.dbname,
            'http_port': self.http_port,
            'root': self.root,
        }

    def client_cmd(self, args=[]):
        return [
            self._get_bin(), 'client',
            '--port=%d' % self.tcp_port,
            '--config-file=%s' % self.config_file,
        ] + args

    def communicate(self, **kwargs):
        def make_command(**kwargs):
            cmdl = self.client_cmd()
            for k, v in kwargs.items():
                cmdl.append('--variable={k}={v}'.format(k=k, v=v))
            return ' '.join(cmdl)

        popen = subprocess.Popen(
            make_command(**kwargs),
            shell=True,
            cwd=self.root,
            env=self._make_env()
        )
        while True:
            try:
                popen.communicate()
            except KeyboardInterrupt:
                # Flush the prompt on CTRL+C rather than quit
                continue
            break
