from __future__ import print_function

import os
from abc import abstractproperty, abstractmethod
from contextlib import contextmanager

import psutil

from mail.devpack.lib import helpers
from mail.devpack.lib.errors import DevpackError
from mail.devpack.lib.helpers import create_root
from mail.devpack.lib.state import read_state
from mail.devpack.lib.yhttp_service import YplatformHttpService
from mail.devpack.lib.yhttp_app_service import YplatformAppHttpService


class AbstractComponent(object):
    DEPS = []

    @abstractproperty
    def NAME(self):
        pass

    @property
    def name(self):
        return self.NAME

    @property
    def root_name(self):
        return self.name

    @abstractproperty
    def config(self):
        pass

    @abstractmethod
    def init_root(self):
        pass

    @abstractmethod
    def start(self):
        pass

    @abstractmethod
    def stop(self):
        pass

    def info(self):
        return {}

    def prepare_data(self):
        pass

    def is_multiroot(self):
        return False

    def communicate(self):
        raise RuntimeError("Component %r can't communicate" % self)

    @staticmethod
    def gen_config(port_generator, config=None):
        return {}

    @contextmanager
    def standalone(self):
        self.init_root()
        try:
            self.start()
            self.prepare_data()
            yield self
        finally:
            self.stop()


class FakeRootComponent(AbstractComponent):
    def __init__(self, env, components):
        self.state = {}
        self.root = create_root(self.name, env.get_config()['sysdata']['root'])

    def start(self):
        pass

    def stop(self):
        pass


class BaseComponent(AbstractComponent):
    def __init__(self, env, components, **kwargs):
        self.env = env
        self.components = components
        self.__state = read_state(self.config, self.name)
        folder_name = self.root_name + ('_' + str(kwargs['port']) if 'port' in kwargs else '')
        self.__root = helpers.create_root(folder_name, self.config.root)
        self.logger = helpers.create_logger(__name__, self, "%s-wrapper.log" % self.name)

    @property
    def config(self):
        return self.env.get_config()

    @property
    def working_dir(self):
        return self.config['sysdata']['root']

    @property
    def state(self):
        return self.__state

    @property
    def root(self):
        return self.__root

    def get_root(self):
        return self.__root

    @property
    def etc_path(self):
        return self.get_etc_path()

    def get_etc_path(self):
        return os.path.join(self.root, 'etc', self.name)

    @property
    def log_dir(self):
        return self.get_log_dir()

    def get_log_dir(self):
        return os.path.join(self.root, 'var', 'log', self.name)

    def info(self):
        return {
            "root": self.root,
            "state": self.state,
        }


class WithPort(AbstractComponent):
    @classmethod
    def gen_config(cls, port_generator, config=None):
        base = super(WithPort, cls).gen_config(port_generator, config=config)
        return dict(
            port=next(port_generator),
            **base
        )

    @property
    def port(self):
        return self.config[self.name]['port']

    @property
    def root_name(self):
        return "%s_%d" % (self.name, self.port)

    def info(self):
        base = super(WithPort, self).info()
        base.update({
            "port": self.port
        })
        return base


class SubprocessComponent(BaseComponent):
    @property
    def log_stdout_path(self):
        return os.path.join(self.root, '%s-stdout.log' % self.name)

    @property
    def log_stderr_path(self):
        return os.path.join(self.root, '%s-stderr.log' % self.name)

    @contextmanager
    def starting(self, cmd, env=None):
        self.logger.info("starting %s, info: %r", self.name, self.info())
        if env is None:
            env = {}
        env.update(os.environ)
        proc = psutil.Popen(
            cmd, cwd=self.root, env=env,
            stdout=open(self.log_stdout_path, 'ab'),
            stderr=open(self.log_stderr_path, 'ab')
        )
        self.state["pid"] = proc.pid
        try:
            # Customization point; one can wait or check process to start up correctly
            yield proc
        except Exception as e:
            self.logger.exception(e)
            for log_name in os.listdir(self.log_dir):
                log_path = os.path.join(self.log_dir, log_name)
                print('log_name: %s' % log_name)
                with open(log_path) as fd:
                    print(fd.read())
            self.stop_proc(proc)
            raise DevpackError("%s does not response to /ping after start" % self.name)
        self.logger.info('%s started, pid: %d', self.name, proc.pid)

    def stop(self):
        self.logger.info("Terminating %s, info: %r", self.name, self.info())
        try:
            proc = self.get_proc()
            if not proc:
                self.logger.warning("No process with pid %s found", self.state["pid"])
            self.stop_proc(proc)
        except psutil.NoSuchProcess:
            self.logger.warning("No process with pid %s found", self.state["pid"])

    def stop_proc(self, proc):
        if not proc:
            return
        proc.terminate()
        try:
            proc.wait(timeout=10)
        except psutil.TimeoutExpired:
            self.logger.warning("Can't wait for termination of process, will try to kill")
            proc.kill()
            try:
                proc.wait(timeout=10)
            except psutil.TimeoutExpired:
                self.logger.error("Can't wait for kill of process, probably manual cleanup is needed")
                raise
        self.logger.info("%s stopped", self.name)

    def get_proc(self):
        try:
            return psutil.Process(self.state["pid"])
        except (KeyError, psutil.NoSuchProcess):
            pass

    def info(self):
        base = super(SubprocessComponent, self).info()
        base.update({
            "proc": self.get_proc()
        })
        return base


class YplatformComponent(BaseComponent):
    @staticmethod
    def gen_config(port_generator, config=None):
        return YplatformHttpService.gen_config(port_generator, config=config)

    def __init__(self, env, components, binary_name, custom_path, **kwargs):
        super(YplatformComponent, self).__init__(env, components, **kwargs)
        self.yhttp = YplatformHttpService(env, self.NAME, binary_name=binary_name, custom_path=custom_path)

    @property
    def etc_path(self):
        return self.yhttp.get_etc_path()

    @abstractproperty
    def service_ticket(self):
        pass

    def stop(self):
        self.yhttp.stop()

    def info(self):
        res = self.yhttp.info()
        res.update({'state': self.state})
        return res

    def get_root(self):
        return self.yhttp.get_root()

    def webserver_port(self):
        return self.yhttp.webserver_port

    def ping(self):
        return self.yhttp.ping()

    def request_get(self, endpoint, **kwargs):
        url = self.yhttp.make_url(endpoint, **kwargs)
        return self.yhttp.get(url, headers={'X-Ya-Service-Ticket': self.service_ticket})

    def request_post(self, endpoint, **kwargs):
        url = self.yhttp.make_url(endpoint, **kwargs)
        return self.yhttp.post(url, data=None, headers={'X-Ya-Service-Ticket': self.service_ticket})


class YplatformAppComponent(BaseComponent):
    @staticmethod
    def gen_config(port_generator, config=None):
        return YplatformAppComponent.gen_config(port_generator, config=config)

    def __init__(self, env, components, custom_path, **kwargs):
        super(YplatformAppComponent, self).__init__(env, components, **kwargs)
        self.yhttp = YplatformAppHttpService(env, self.NAME, custom_path=custom_path)

    @abstractproperty
    def service_ticket(self):
        pass

    def start(self, put_pgpass_in_env=False):
        if put_pgpass_in_env:
            self.yhttp.put_pgpassfile_in_env(self.get_root())

        self.yhttp.tweak_env_for_asan()
        self.yhttp.start('pong')

    def stop(self):
        self.yhttp.stop()

    def info(self):
        res = self.yhttp.info()
        res.update({'state': self.state})
        return res

    def get_root(self):
        return self.yhttp.get_root()

    def webserver_port(self):
        return self.yhttp.webserver_port

    def ping(self):
        return self.yhttp.ping()

    def request_get(self, endpoint, **kwargs):
        url = self.yhttp.make_url(endpoint, **kwargs)
        return self.yhttp.get(url, headers={'X-Ya-Service-Ticket': self.service_ticket})

    def request_post(self, endpoint, **kwargs):
        url = self.yhttp.make_url(endpoint, **kwargs)
        return self.yhttp.post(url, data=None, headers={'X-Ya-Service-Ticket': self.service_ticket})

    @property
    def config_path(self):
        return self.yhttp.get_config_path()

    @property
    def secrets_path(self):
        return self.yhttp.get_secrets_path()

    @property
    def resources_path(self):
        return self.yhttp.get_resources_path()


class DogComponent(YplatformComponent):
    def __init__(self, env, components, binary_name, custom_path, **kwargs):
        port = env.get_config()[self.name]['webserver_port']
        super(DogComponent, self).__init__(env, components, binary_name=binary_name, custom_path=custom_path, port=port, **kwargs)

    def start(self):
        self.yhttp.start('pong')


class DogAppComponent(YplatformAppComponent):
    def __init__(self, env, components, custom_path, **kwargs):
        port = env.get_config()[self.name]['webserver_port']
        super(DogAppComponent, self).__init__(env, components, binary_name='application', custom_path=custom_path, port=port, **kwargs)

    @staticmethod
    def gen_config(port_generator, config=None):
        return YplatformAppHttpService.gen_config(port_generator, config=config)
