import logging
import sqlparse
import re

import library.python.resource as rs


from mail.devpack.lib.ch import ClickHouse as ClickHouseLib
from mail.devpack.lib.state import read_state, INIT_STATE
from mail.devpack.lib.components.base import AbstractComponent
from mail.devpack.lib.db import DbComponentMixin


log = logging.getLogger(__name__)


class ClickHouse(AbstractComponent, DbComponentMixin):
    DEPS = []

    @classmethod
    def gen_config(cls, port_generator, config=None):
        return {
            'http_port': next(port_generator),
            'tcp_port': next(port_generator),
        }

    def __init__(self, dbname, http_port, tcp_port, users, ddl_prefixes, root=None, state=None):
        self._init(dbname, http_port, tcp_port, users, ddl_prefixes, root, state)

    def init_from_conf(self, config, dbname, users, ddl_prefixes=None):
        if ddl_prefixes is None:
            ddl_prefixes = self.before_all_prefixes + self.snapshot_sql_files + self.after_all_prefixes
        self._init(
            dbname=dbname,
            http_port=config[self.NAME]['http_port'],
            tcp_port=config[self.NAME]['tcp_port'],
            users=users,
            ddl_prefixes=ddl_prefixes,
            root=config.root,
            state=read_state(config, self.NAME),
        )

    def _init(self, dbname, http_port, tcp_port, users, ddl_prefixes, root, state):
        self.__state = state or INIT_STATE.copy()
        self.ch = ClickHouseLib(dbname, http_port, tcp_port=tcp_port, working_dir=root)
        self.users = users
        self.ddl_prefixes = ddl_prefixes

    @property
    def NAME(self):
        return self.ch.dbname

    @property
    def state(self):
        return self.__state

    def init_root(self):
        self.ch.extract_tar()

    def get_root(self):
        return self.ch.root

    def port(self):
        return self.ch.http_port

    def start(self):
        return self.ch.start()

    def stop(self):
        return self.ch.stop()

    def restart(self):
        self.stop()
        return self.start()

    def prepare_data(self):
        self.ch.dropdb()
        self.ch.createdb()
        self._create_users()
        self._apply_migrations()

    def execute(self, query, **kwargs):
        return self.ch.execute(query, **kwargs)

    def query(self, query, **kwargs):
        return self.ch.query(query, **kwargs)

    def info(self):
        res = self.ch.info()
        res.update({'state': self.state})
        return res

    def communicate(self, **kwargs):
        return self.ch.communicate(**kwargs)

    def _create_users(self):
        for user in self.users:
            self.execute('drop user if exists %s' % user)
            self.execute('create user %s' % user)

    def _apply_migrations(self):
        for ddl_prefix in self.ddl_prefixes:
            apply_resources_sql(self.ch, ddl_prefix)


def convert_cluster_query_to_local(query):
    query = re.sub('ON\\s+CLUSTER\\s+\'{cluster}\'', '', query, flags=re.IGNORECASE)
    query = re.sub(r'Replicated([a-z]+)\(.*\)', r'\1', query, flags=re.IGNORECASE)
    return query


def apply_resources_sql(db, prefix):
    for path, query_text in rs.iteritems(prefix=prefix):
        try:
            log.info('apply %s %s', path, query_text)
            query_text = query_text.decode('utf-8')

            query_text = convert_cluster_query_to_local(query_text)

            for subquery in sqlparse.split(query_text):
                if subquery.strip():
                    db.execute(subquery)
        except Exception as e:
            raise RuntimeError('%s\n%s\n%s\n%s' % (prefix, path, query_text, e))
