import logging

import sqlparse
import library.python.resource as rs

from mail.devpack.lib.pg import Postgresql
from mail.devpack.lib.db import DbComponentMixin
from mail.devpack.lib.state import read_state, INIT_STATE
from mail.devpack.lib.components.base import AbstractComponent

log = logging.getLogger(__name__)


class Postgres(AbstractComponent, DbComponentMixin):
    DEPS = []

    @classmethod
    def gen_config(cls, port_generator, config=None):
        return {"port": next(port_generator)}

    def __init__(self, dbname, port, users, ddl_prefixes, root=None, state=None):
        self._init(dbname, port, users, ddl_prefixes, root, state)

    def init_from_conf(self, config, dbname, users, ddl_prefixes=None):
        if ddl_prefixes is None:
            ddl_prefixes = self.before_all_prefixes + self.snapshot_sql_files + self.after_all_prefixes
        self._init(
            dbname=dbname,
            port=config[self.NAME]['port'],
            users=users,
            ddl_prefixes=ddl_prefixes,
            root=config.root,
            state=read_state(config, self.NAME),
        )

    def _init(self, dbname, port, users, ddl_prefixes, root, state):
        self.__state = state or INIT_STATE.copy()
        self.pg = Postgresql(port, dbname, root)
        self.users = users
        self.ddl_prefixes = ddl_prefixes

    @property
    def NAME(self):
        return self.pg.dbname

    @property
    def state(self):
        return self.__state

    def init_root(self):
        self.pg.extract_tar()
        self.pg.initdb()

    def get_root(self):
        return self.pg.root

    def port(self):
        return self.pg.port

    def start(self):
        return self.pg.start()

    def stop(self):
        return self.pg.stop()

    def restart(self):
        self.stop()
        return self.start()

    def prepare_data(self):
        self.pg.dropdb()
        self.pg.createdb()
        self._create_users()
        self._apply_migrations()

    def execute(self, query, **kwargs):
        return self.pg.execute(query, **kwargs)

    def query(self, query, **kwargs):
        return self.pg.query(query, **kwargs)

    def info(self):
        res = self.pg.info()
        res.update({'state': self.state})
        return res

    def dsn(self):
        return self.pg.dsn()

    def communicate(self, **kwargs):
        return self.pg.communicate(**kwargs)

    def _create_users(self):
        self.execute('create user root superuser createdb inherit login;')
        for user in self.users:
            self.execute('drop user if exists %s' % user)
            self.execute('create user %s' % user)

    def _apply_migrations(self):
        for ddl_prefix in self.ddl_prefixes:
            apply_resources_sql(self.pg, ddl_prefix)


def apply_resources_sql(pg, prefix):
    for path, query_text in rs.iteritems(prefix=prefix):
        try:
            log.info('apply %s %s', path, query_text)
            if 'EMPTY' in path:
                continue
            if 'NONTRANSACTIONAL' in path:
                for subquery in sqlparse.split(query_text):
                    if subquery.strip():
                        pg.execute(subquery)
            else:
                if query_text.strip():
                    pg.execute(query_text)
        except Exception as e:
            raise RuntimeError('%s\n%s\n%s\n%s' % (prefix, path, query_text, e))
