import os
import time
import backoff
import requests
import socket

from library.python import resource
from mail.devpack.lib import helpers
from mail.devpack.lib.state import read_state
from mail.devpack.lib.errors import DevpackError
from mail.devpack.lib.yhttp_service import YplatformHttpService
from mail.devpack.lib.components.base import AbstractComponent
from mail.devpack.lib.components.cloud_cluster import CloudCluster

from .sharddb import ShardDb
from .mdb import Mdb
from .fakebb import FakeBlackbox
from .tvmapi import TvmApi

from abc import abstractmethod


class BaseApiSpecificSettingsProvider(object):
    @abstractmethod
    def get_api_cls(self):
        pass

    @abstractmethod
    def get_devpack_config_path(self):
        pass


class CloudApiSpecificSettingsProvider(object):
    def get_api_cls(self):
        return SharpeiCloudApi

    def get_devpack_config_path(self):
        return 'sharpei/config-devpack-cloud.yml'


class DataSyncApiSpecificSettingsProvider(BaseApiSpecificSettingsProvider):
    def get_api_cls(self):
        return SharpeiDiskAndDatasyncApi

    def get_devpack_config_path(self):
        return 'sharpei/config-devpack-datasync.yml'


class DiskApiSpecificSettingsProvider(BaseApiSpecificSettingsProvider):
    def get_api_cls(self):
        return SharpeiDiskAndDatasyncApi

    def get_devpack_config_path(self):
        return 'sharpei/config-devpack-disk.yml'


class MailApiSpecificSettingsProvider(BaseApiSpecificSettingsProvider):
    def __init__(self, kind):
        self.__kind = kind

    def get_api_cls(self):
        return SharpeiApi

    def get_devpack_config_path(self):
        return 'sharpei/config-devpack-{kind}.yml'.format(kind=self.__kind)


def ApiSpecificSettingsProviderFactory(kind):
    if kind == 'mail':
        return MailApiSpecificSettingsProvider(kind)
    elif kind == 'disk':
        return DiskApiSpecificSettingsProvider()
    elif kind == 'datasync':
        return DataSyncApiSpecificSettingsProvider()
    elif kind == 'cloud':
        return CloudApiSpecificSettingsProvider()


class BaseSharpei(AbstractComponent):
    @staticmethod
    def gen_config(port_generator, config=None):
        return YplatformHttpService.gen_config(port_generator)

    def __init__(self, env):
        self.__state = read_state(env.get_config(), self.name)
        self.__service = YplatformHttpService(
            env=env,
            name=self.name,
            binary_name='sharpei',
            custom_path='sharpei'
        )
        self.__kind = env.get_config().get('kind', 'mail')
        self.__settings_provider = ApiSpecificSettingsProviderFactory(self.__kind)

    @property
    def state(self):
        return self.__state

    @abstractmethod
    def format_config(self, config, service):
        pass

    @abstractmethod
    def write_configs(self, etc_path):
        pass

    def init_root(self):
        self.__service.init_root()

        etc_path = self.__service.get_etc_path()
        self.write_configs(etc_path)
        helpers.write2file(resource.find('sharpei/tvm_secret'), os.path.join(etc_path, 'tvm_secret'))

        config = resource.find(self.__settings_provider.get_devpack_config_path())
        config = self.format_config(config, self.__service)
        helpers.write2file(config, self.__service.get_config_path())

    def get_root(self):
        return self.__service.get_root()

    def start(self):
        self.__service.start("pong")

    def stop(self):
        self.__service.stop()

    def restart(self):
        self.stop()
        self.start()

    def prepare_data(self):
        api = self.api()
        wait_sharpei_cache(api)
        api.conninfo(uid=1000, mode='write_only')

    def info(self):
        res = self.__service.info()
        res.update({"state": self.state})
        return res

    def webserver_port(self):
        return self.__service.webserver_port

    def ping(self):
        return self.__service.ping()

    def api(self):
        api = self.__settings_provider.get_api_cls()
        return api(location='http://[::1]:%d' % self.webserver_port())


def wait_sharpei_cache(sharpei_api):
    retries = 30
    for _ in range(retries):
        response = sharpei_api.stat()
        if response.status_code == 200 and response.text != '{}':
            return True
        time.sleep(1)
    raise DevpackError("sharpei expects some predefined instances in sharddb, but cannot read them and output in /stat")


class Sharpei(BaseSharpei):
    NAME = 'sharpei'
    DEPS = [ShardDb, Mdb, FakeBlackbox, TvmApi]

    def format_config(self, config, service):
        return service.format_config(
            config,
            sharddb_port=self.__sharddb_port,
            bb_port=self.__blackbox_port,
            tvm_api_port=self.__tvm_api_port,
            hostname=socket.gethostname(),
        )

    def write_configs(self, etc_path):
        for cfg_name in ('config-base.yml', 'config-mail-base.yml', 'config-disk.yml', 'config-datasync.yml'):
            helpers.write2file(resource.find(os.path.join("sharpei", cfg_name)), os.path.join(etc_path, cfg_name))

    def __init__(self, env, components):
        super(Sharpei, self).__init__(
            env=env,
        )
        self.__sharddb_port = components[ShardDb].port()
        self.__tvm_api_port = components[TvmApi].port
        self.__blackbox_port = components[FakeBlackbox].port


class SharpeiWithBlackboxMock(BaseSharpei):
    NAME = 'sharpei_with_blackbox_mock'
    DEPS = [ShardDb, Mdb, TvmApi]

    @staticmethod
    def gen_config(port_generator, config=None):
        config = YplatformHttpService.gen_config(port_generator)
        config['pyremock_port'] = next(port_generator)
        return config

    def format_config(self, config, service):
        return service.format_config(
            config,
            sharddb_port=self.__sharddb_port,
            bb_port=self.__blackbox_port,
            tvm_api_port=self.__tvm_api_port,
            hostname=socket.gethostname(),
        )

    def write_configs(self, etc_path):
        for cfg_name in ('config-base.yml', 'config-mail-base.yml', 'config-disk.yml', 'config-datasync.yml'):
            helpers.write2file(resource.find(os.path.join("sharpei", cfg_name)), os.path.join(etc_path, cfg_name))

    def __init__(self, env, components):
        pyremock_port = env.get_config()[self.NAME]['pyremock_port']
        super(SharpeiWithBlackboxMock, self).__init__(
            env=env,
        )
        self.__pyremock_port = pyremock_port
        self.__sharddb_port = components[ShardDb].port()
        self.__tvm_api_port = components[TvmApi].port
        self.__blackbox_port = pyremock_port

    def prepare_data(self):
        wait_sharpei_cache(self.api())

    def pyremock_port(self):
        return self.__pyremock_port


class SharpeiCloud(BaseSharpei):
    NAME = 'sharpei_cloud'
    DEPS = [CloudCluster]

    @staticmethod
    def gen_config(port_generator, config=None):
        config = YplatformHttpService.gen_config(port_generator)
        config['iam_port'] = next(port_generator)
        config['yc_port'] = next(port_generator)
        return config

    def format_config(self, config, service):
        return service.format_config(
            config,
            iam_port=self.__iam_port,
            yc_port=self.__yc_port,
            cluster_host_port=self.__cluster_host_port,
            etc_path=service.get_etc_path(),
        )

    def write_configs(self, etc_path):
        helpers.write2file(resource.find("sharpei/config-cloud.yml"), os.path.join(etc_path, 'config-cloud.yml'))
        helpers.write2file(resource.find("sharpei/jwt"), os.path.join(etc_path, "jwt"))

    def __init__(self, env, components):
        super(SharpeiCloud, self).__init__(
            env=env,
        )
        cfg = env.get_config()
        self.__iam_port = cfg[self.NAME]['iam_port']
        self.__yc_port = cfg[self.NAME]['yc_port']
        assert len(cfg['cloud_cluster']['shards']) == 1 and len(cfg['cloud_cluster']['shards'][0]['dbs']) == 1
        # assert is needed, because you can't configure the service to have multiple databases on the same host.
        # TODO: delete assert when it becomes possible to configure the service differently
        self.__cluster_host_port = cfg["cloud_cluster"]['shards'][0]['dbs'][0]['port']

    def prepare_data(self):
        wait_sharpei_cache(self.api())

    def iam_port(self):
        return self.__iam_port

    def yc_port(self):
        return self.__yc_port


class SharpeiCloudApi(object):
    def __init__(self, location='http://localhost:9999'):
        self.location = location

    @backoff.on_exception(backoff.expo, requests.HTTPError, max_time=1)
    def ping(self, request_id='devpack'):
        return requests.get(
            self.location + '/ping',
            timeout=10,
            headers=self.make_headers(request_id),
        )

    @backoff.on_exception(backoff.expo, requests.HTTPError, max_time=1)
    def stat(self, request_id='devpack'):
        return requests.get(
            self.location + '/stat',
            timeout=10,
            headers=self.make_headers(request_id),
        )

    @backoff.on_exception(backoff.expo, requests.HTTPError, max_time=1)
    def stat_v3(self, request_id='devpack', shard_id=None):
        location = self.location + '/v3/stat'
        if shard_id:
            location += '?shard_id=' + str(shard_id)
        return requests.get(
            location,
            timeout=10,
            headers=self.make_headers(request_id),
        )

    @staticmethod
    def make_headers(request_id):
        return {'X-Request-Id': str(request_id)}


class SharpeiApi(object):
    def __init__(self, location='http://localhost:9999'):
        self.location = location

    @backoff.on_exception(backoff.expo, requests.HTTPError, max_time=1)
    def ping(self, request_id='devpack'):
        return requests.get(
            self.location + '/ping',
            timeout=10,
            headers=self.make_headers(request_id),
        )

    @backoff.on_exception(backoff.expo, requests.HTTPError, max_time=1)
    def pingdb(self, request_id='devpack'):
        return requests.get(
            self.location + '/pingdb',
            timeout=10,
            headers=self.make_headers(request_id),
        )

    @backoff.on_exception(backoff.expo, requests.HTTPError, max_time=1)
    def stat(self, request_id='devpack'):
        return requests.get(
            self.location + '/stat',
            timeout=10,
            headers=self.make_headers(request_id),
        )

    @backoff.on_exception(backoff.expo, requests.HTTPError, max_time=1)
    def stat_v2(self, request_id='devpack', shard_id=None):
        location = self.location + '/v2/stat'
        if shard_id:
            location += '?shard_id=' + str(shard_id)
        return requests.get(
            location,
            timeout=10,
            headers=self.make_headers(request_id),
        )

    @backoff.on_exception(backoff.expo, requests.HTTPError, max_time=1)
    def stat_v3(self, request_id='devpack', shard_id=None):
        location = self.location + '/v3/stat'
        if shard_id:
            location += '?shard_id=' + str(shard_id)
        return requests.get(
            location,
            timeout=10,
            headers=self.make_headers(request_id),
        )

    @backoff.on_exception(backoff.expo, requests.HTTPError, max_time=1)
    def sharddb_stat(self, request_id='devpack'):
        location = self.location + '/sharddb_stat'
        return requests.get(
            location,
            timeout=10,
            headers=self.make_headers(request_id),
        )

    @backoff.on_exception(backoff.expo, requests.HTTPError, max_time=1)
    def conninfo(self, uid, mode, force=None, request_id='devpack'):
        location = self.location + '/conninfo?uid={uid}&mode={mode}'.format(uid=uid, mode=mode)
        if force:
            location += '&force={force}'.format(force=force)
        return requests.get(
            location,
            timeout=10,
            headers=self.make_headers(request_id),
        )

    @backoff.on_exception(backoff.expo, requests.HTTPError, max_time=1)
    def domain_conninfo(self, domain_id, request_id='devpack'):
        return requests.get(
            self.location + '/domain_conninfo?domain_id={domain_id}'.format(domain_id=domain_id),
            timeout=10,
            headers=self.make_headers(request_id),
        )

    @backoff.on_exception(backoff.expo, requests.HTTPError, max_time=1)
    def org_conninfo(self, org_id, request_id='devpack'):
        return requests.get(
            self.location + '/org_conninfo?org_id={org_id}'.format(org_id=org_id),
            timeout=10,
            headers=self.make_headers(request_id),
        )

    @backoff.on_exception(backoff.expo, requests.HTTPError, max_time=1)
    def deleted_conninfo(self, uid, mode, request_id='devpack'):
        location = self.location + '/deleted_conninfo?uid={uid}&mode={mode}'.format(uid=uid, mode=mode)
        return requests.get(
            location,
            timeout=10,
            headers=self.make_headers(request_id),
        )

    @backoff.on_exception(backoff.expo, requests.HTTPError, max_time=1)
    def reset(self, shard, request_id='devpack'):
        return requests.post(
            self.location + '/reset?shard={shard}'.format(shard=shard),
            timeout=10,
            headers=self.make_headers(request_id),
        )

    @staticmethod
    def make_headers(request_id):
        return {'X-Request-Id': str(request_id)}


class SharpeiDiskAndDatasyncApi(object):
    '''
    Disk and Datasync share the same api except that Datasync allows uid to be textual.
    '''
    def __init__(self, location='http://localhost:9999'):
        self.location = location

    @backoff.on_exception(backoff.expo, requests.HTTPError, max_time=1)
    def ping(self, request_id='devpack'):
        return requests.get(
            self.location + '/ping',
            timeout=10,
            headers=self.make_headers(request_id),
        )

    @backoff.on_exception(backoff.expo, requests.HTTPError, max_time=1)
    def pingdb(self, request_id='devpack'):
        return requests.get(
            self.location + '/pingdb',
            timeout=10,
            headers=self.make_headers(request_id),
        )

    @backoff.on_exception(backoff.expo, requests.HTTPError, max_time=1)
    def stat(self, request_id='devpack', shard_id=None):
        location = self.location + '/stat'
        if shard_id:
            location += '?shard_id=' + str(shard_id)
        return requests.get(
            location,
            timeout=10,
            headers=self.make_headers(request_id),
        )

    @backoff.on_exception(backoff.expo, requests.HTTPError, max_time=1)
    def get_user(self, uid, mode, force=None, request_id='devpack'):
        location = self.location + '/get_user?uid={uid}&mode={mode}'.format(uid=uid, mode=mode)
        if force:
            location += '&force={force}'.format(force=force)
        return requests.get(
            location,
            timeout=10,
            headers=self.make_headers(request_id),
        )

    @backoff.on_exception(backoff.expo, requests.HTTPError, max_time=1)
    def create_user(self, uid, shard_id=None, request_id='devpack'):
        location = self.location + '/create_user?uid={uid}'.format(uid=uid)
        if shard_id:
            location += '&shard_id={shard_id}'.format(shard_id=shard_id)
        return requests.post(
            location,
            timeout=10,
            headers=self.make_headers(request_id),
        )

    @backoff.on_exception(backoff.expo, requests.HTTPError, max_time=1)
    def update_user(self, uid, shard_id=None, new_shard_id=None, data=None, request_id='devpack'):
        location = self.location + '/update_user?uid={uid}'.format(uid=uid)
        if shard_id:
            location += '&shard_id={shard_id}'.format(shard_id=shard_id)
        if new_shard_id:
            location += '&new_shard_id={new_shard_id}'.format(new_shard_id=new_shard_id)
        return requests.post(
            location,
            json=data,
            timeout=10,
            headers=self.make_headers(request_id),
        )

    @backoff.on_exception(backoff.expo, requests.HTTPError, max_time=1)
    def reset_cache(self, shard, request_id='devpack'):
        return requests.post(
            self.location + '/reset_cache?shard={shard}'.format(shard=shard),
            timeout=10,
            headers=self.make_headers(request_id),
        )

    @staticmethod
    def make_headers(request_id):
        return {'X-Request-Id': str(request_id)}
