import os.path
import requests
import retrying

from library.python import resource
from mail.devpack.lib import helpers
from mail.devpack.lib.state import read_state
from mail.webmail_config.lib.make_config import make_config
import yaml
from mail.devpack.lib.yhttp_service import YplatformHttpService
from mail.devpack.lib.components.base import BaseComponent

from .tvmapi import TvmApi


class Sheltie(BaseComponent):
    NAME = 'sheltie'
    DEPS = [TvmApi]

    def __init__(self, env, components):
        config = env.get_config()
        self.__components = components
        self.__state = read_state(config=config, component_name=self.NAME)
        self.__service = YplatformHttpService(
            env=env,
            name=self.NAME,
            binary_name='sheltie',
            custom_path='sheltie'
        )
        self.__pyremock_port = config[self.NAME]['pyremock_port']

    @property
    def state(self):
        return self.__state

    @staticmethod
    def gen_config(port_generator, config=None):
        result = YplatformHttpService.gen_config(port_generator, config=config)
        result['pyremock_port'] = next(port_generator)
        return result

    def init_root(self):
        self.__service.init_root()
        self._generate_config('development')
        helpers.write2file(
            what=resource.find('sheltie/tvm_secret'),
            path=os.path.join(self.__service.get_secrets_path(), 'tvm_secret'),
        )
        helpers.write2file(
            what=self.__service.format_config(
                config=resource.find('sheltie/config-devpack.yml'),
                tvm_api_port=self.__components[TvmApi].port,
                pyremock_port=self.__pyremock_port,
            ),
            path=self.__service.get_config_path(),
        )
        python_path = os.path.join(self.__service.get_app_path(), 'python')
        helpers.mkdir_recursive(python_path)
        helpers.write2file(
            what=resource.find('sheltie/transform.py'),
            path=os.path.join(python_path, 'transform.py'),
        )

    def _generate_config(self, env):
        base = resource.find('sheltie/base.yml')
        service_yaml = yaml.safe_load(resource.find('sheltie/service.yaml'))
        common_yaml = yaml.safe_load(resource.find('webmail_config/common.yaml'))
        cfg = make_config(env, base, service_yaml, common_yaml, silent=True)
        config = self.__service.format_config(cfg)
        helpers.write2file(config, os.path.join(self.__service.get_etc_path(), 'sheltie.yml'))

    def get_root(self):
        return self.__service.get_root()

    def start(self):
        self.__service.start('pong')

    def stop(self):
        self.__service.stop()

    def info(self):
        return dict(state=self.__state, **self.__service.info())

    def prepare_data(self):
        pass

    def webserver_port(self):
        return self.__service.webserver_port

    def pyremock_port(self):
        return self.__pyremock_port

    def ping(self):
        return self.__service.ping()

    def api(self, uid=None, service_ticket=None, user_ticket=None):
        return SheltieApi(
            location='http://localhost:%d' % self.webserver_port(),
            uid=uid,
            service_ticket=service_ticket,
            user_ticket=user_ticket,
        )

    def get_config_path(self):
        return self.__service.get_config_path()


class SheltieApi(object):
    def __init__(self, location='http://localhost:4080', uid=None, service_ticket=None, user_ticket=None):
        self.location = location
        self.uid = uid
        self.service_ticket = service_ticket
        self.user_ticket = user_ticket

    @retrying.retry(stop_max_delay=10000)
    def ping(self, request_id):
        return requests.get(
            self.location + '/ping',
            headers=self.make_headers(request_id),
            timeout=1,
        )

    @retrying.retry(stop_max_delay=10000)
    def colabook_import(self, request_id, body):
        return requests.post(
            self.location + '/compat/colabook_import.json?uid={uid}&format=json'.format(uid=self.uid),
            headers=self.make_headers(request_id),
            data=body,
            timeout=5
        )

    @retrying.retry(stop_max_delay=10000)
    def colabook_export(self, request_id):
        return requests.get(
            self.location + '/compat/colabook_export?uid={uid}&format=json'.format(uid=self.uid),
            headers=self.make_headers(request_id),
            timeout=5
        )

    @retrying.retry(stop_max_delay=10000)
    def to_vcard(self, request_id, body):
        return requests.post(
            self.location + '/v1/to_vcard?uid={uid}'.format(uid=self.uid),
            headers=self.make_headers(request_id),
            data=body,
            timeout=5
        )

    @retrying.retry(stop_max_delay=10000)
    def from_vcard(self, request_id, body):
        return requests.post(
            self.location + '/v1/from_vcard?uid={uid}'.format(uid=self.uid),
            headers=self.make_headers(request_id),
            data=body,
            timeout=5
        )

    def make_headers(self, request_id):
        return {
            'X-Request-Id': request_id,
        }
