from werkzeug import routing
from werkzeug.exceptions import HTTPException, NotFound
from werkzeug.routing import Rule
from werkzeug.wrappers import Request, Response
from .werkzeug_component import WerkzeugComponent
from .mulcagate import Mulcagate
from .tvmtool import TvmTool


class TvmMulcaApp(object):
    url_map = routing.Map([
        Rule('/', endpoint='describe'),
        Rule('/_list', endpoint='list'),

        Rule('/gate/put/<suid>', endpoint='put', methods=['POST']),
        Rule('/gate/get/<stid>', endpoint='get', methods=['GET']),
        Rule('/gate/del/<stid>', endpoint='del', methods=['GET']),
    ])

    def __init__(self, mulca_comp, tvm_comp, tvm_id):
        self.tvm_comp = tvm_comp
        self.mulca_comp = mulca_comp
        self.tvm_id = tvm_id

    def check_tvm(func):
        def func_modified(self, request, *args, **kwargs):
            tvm_header_name = "X-Ya-Service-Ticket"
            if tvm_header_name not in request.headers.keys():
                return Response("No TVM ticket provided", status=403)
            try:
                is_valid = self.tvm_comp.is_ticket_valid(request.headers[tvm_header_name], self.tvm_id)
            except BaseException as exc:
                return Response("Error while checking ticket: %r" % exc, status=500)
            if is_valid:
                return func(self, request, *args, **kwargs)
            else:
                return Response("Wrong TVM ticket", status=403)

        return func_modified

    def on_describe(self, request):
        return self.mulca_comp.describe()

    @check_tvm
    def on_list(self, request):
        return self.mulca_comp.list()

    @check_tvm
    def on_get(self, request, stid):
        return self.mulca_comp.get(stid)

    @check_tvm
    def on_put(self, request, suid):
        return self.mulca_comp.put(suid, request.get_data(as_text=True))

    @check_tvm
    def on_del(self, request, stid):
        return self.mulca_comp.delete(stid)

    def dispatch_request(self, request):
        adapter = self.url_map.bind_to_environ(request.environ)
        try:
            endpoint, values = adapter.match()
            return getattr(self, 'on_' + endpoint)(request, **values)
        except NotFound:
            return Response('Endpoint not found', status=500)
        except HTTPException as e:
            return e

    def __call__(self, environ, start_response):
        request = Request(environ)
        response = self.dispatch_request(request)
        if not isinstance(response, Response):
            response = Response(
                response.text,
                status=response.status_code
            )
        return response(environ, start_response)


class TvmMulcagate(WerkzeugComponent):
    NAME = "tvm_mulcagate"
    DEPS = [Mulcagate, TvmTool]

    def __init__(self, env, components):
        app = TvmMulcaApp(
            mulca_comp=components[Mulcagate],
            tvm_comp=components[TvmTool],
            tvm_id=env.config['tvm_mulcagate']['tvm']['client_id'],
        )
        self.tvm_id = env.config['tvm_mulcagate']['tvm']['client_id']
        super(TvmMulcagate, self).__init__(env, components, app=app, is_https=True)

    def put(self, stid, data):
        pass

    def get(self, stid):
        pass

    def delete(self, stid):
        pass

    @classmethod
    def gen_config(cls, port_generator, config=None):
        base = super(TvmMulcagate, cls).gen_config(port_generator, config=config)
        return dict(
            tvm={"client_id": next(port_generator)},
            **base
        )
