import os

import requests
import yaml
from werkzeug import exceptions as http_exceptions
from werkzeug.formparser import parse_form_data
from werkzeug.wrappers import Response, Request
from werkzeug.wsgi import responder

import json
from library.python import resource
from mail.devpack.lib import helpers
from mail.devpack.lib.components.base import BaseComponent, WithPort
from mail.devpack.lib.errors import DevpackError


class TvmApiApp(object):
    def __init__(self, public_keys, service_tickets, logger):
        self.logger = logger
        self.public_keys = public_keys
        self.service_tickets = service_tickets

    def ping(self, request):
        return Response('ok', mimetype='text/plain')

    def keys(self, request):
        return Response(self.public_keys, mimetype='text/plain')

    def _make_error_resp(self, name):
        return Response(status=400, response=json.dumps({
            "status": "ERR_REQUEST",
            "error": "MISSING." + name,
            "desc": "Arg is required but empty: " + name
        }))

    def tickets(self, request):
        _, args, _ = parse_form_data(request.environ)
        if 'src' not in args:
            return self._make_error_resp('src')

        if 'dst' not in args:
            return self._make_error_resp('dst')

        src = args['src']
        dsts = map(lambda dst: dst, args['dst'].split(','))

        ret = {}
        for dst in dsts:
            ret[dst] = {
                'ticket': self.service_tickets[src][dst]
            }

        return Response(json.dumps(ret), mimetype='application/json')

    @responder
    def __call__(self, environ, start_response):
        request = Request(environ)
        handlers = {
            u'/ping': self.ping,
            u'/2/keys': self.keys,
            u'/2/keys/': self.keys,
            u'/2/ticket': self.tickets,
            u'/2/ticket/': self.tickets,
        }

        for path, h in handlers.items():
            if request.path.startswith(path):
                return h(request)
        return http_exceptions.NotFound()


class TvmApi(BaseComponent, WithPort):
    NAME = "tvmapi"
    DEPS = []

    def __init__(self, env, components):
        super(TvmApi, self).__init__(env, components)

        self.public_keys = resource.find('tvmapi/keys')
        self.service_tickets = {}
        self.service_tickets_by_name = {}

        for source in yaml.safe_load(resource.find('tvmapi/service_tickets')):
            dsts = {}
            names = {}
            for dst in source['dsts']:
                dsts[dst['id']] = dst['ticket']
                names[dst['name']] = dst['ticket']
            self.service_tickets[source['src']] = dsts
            self.service_tickets_by_name[source['name']] = names

        self.app = TvmApiApp(self.public_keys, self.service_tickets, self.logger)

    def _request(self, query, timeout=10):
        url = "http://localhost:{port}{query}".format(port=self.port, query=query)
        self.logger.info('http call: %s', url)
        return requests.get(url, timeout=timeout)

    def start(self):
        self.logger.info('starting tvmapi on port %d', self.port)
        child_pid = os.fork()
        if child_pid:
            try:
                helpers.wait_ping(self.logger, self.ping)
            except:
                raise DevpackError("tvmapi does not response to /ping after start")
            self.state["pid"] = child_pid
            self.logger.info('tvmapi started, pid: %d', child_pid)
        else:
            helpers.start_werkzeug(self)

    def stop(self):
        pid = self.state.get("pid")
        if pid:
            self.logger.info('stopping tvmapi on port %d with pid %d', self.port, pid)
            helpers.kill_proc(pid)
            del self.state["pid"]
            self.logger.info('tvmapi stopped')

    def info(self):
        return {
            "state": self.state,
            "root": self.root,
            "port": self.port,
        }

    def ping(self):
        return self._request('/ping')

    def get_ticket(self, src_name, dst_name):
        return self.service_tickets_by_name[src_name][dst_name]
