import os
import stat

import requests

import yaml
import json
from library.python import resource
from mail.devpack.lib import helpers
from mail.devpack.lib.components.base import WithPort
from .base import SubprocessComponent


class TvmTool(SubprocessComponent, WithPort):
    NAME = "tvmtool"
    AUTHTOKEN = '{:Z<32}'.format('TVMTOOL-AUTHTOKEN-')  # Must be 32 bytes long

    @classmethod
    def gen_config(cls, port_generator, config=None):
        if config:
            config['cron'] = {'tvm': {'client_id': next(port_generator)}}
        base = super(TvmTool, cls).gen_config(port_generator, config=config)
        return dict(
            auth=TvmTool.AUTHTOKEN,
            **base
        )

    def __init__(self, env, components):
        super(TvmTool, self).__init__(env, components)
        self.bin_file = os.path.join(self.root, 'tvmtool')
        self.cron_id = env.config['cron']['tvm']['client_id']

    def get_config_path(self):
        return os.path.join(self.root, 'config.yaml')

    def generate_tvmtool_config(self):
        comps = [(name, conf) for name, conf in self.config.items() if conf and 'tvm' in conf]
        cfg = {
            "BbEnvType": 1,
            "clients": {
                comp_name: {
                    "self_tvm_id": comp_conf['tvm']['client_id'],
                    "dsts": {
                        dst_comp_name: {"dst_id": dst_comp_conf['tvm']['client_id']}
                        for dst_comp_name, dst_comp_conf in comps
                    }
                }
                for comp_name, comp_conf in comps
            }
        }
        cfg['clients'].update({
            source['name']: {
                "self_tvm_id": int(source['src']),
                "dsts": {
                    dsts['name']: {"dst_id": int(dsts['id'])}
                    for dsts in source['dsts']
                }
            }
            for source in yaml.safe_load(resource.find('tvmapi/service_tickets'))})

        return json.dumps(cfg, indent=4)

    def init_root(self):
        helpers.write2file(resource.find('tvmtool'), self.bin_file)
        os.chmod(self.bin_file, os.stat(self.bin_file).st_mode | stat.S_IEXEC)
        config = self.generate_tvmtool_config()
        helpers.write2file(config, self.get_config_path())

    def start(self, ping_response='OK'):
        cmd = [self.bin_file, '--port', str(self.port), '-c', self.get_config_path(), '-v', '--unittest']
        env = {'QLOUD_TVM_TOKEN': self.AUTHTOKEN}
        with self.starting(cmd, env):
            helpers.wait_ping(self.logger, self.ping, ping_text=ping_response)

    def info(self):
        return {
            "root": self.root,
            "state": self.state,
            "bin_file": self.bin_file,
            "port": self.port,
            "tvm_config": self.generate_tvmtool_config(),
        }

    def get(self, query, timeout=10, **kwargs):
        url = "http://localhost:{port}{query}".format(port=self.port, query=query)
        self.logger.info('GET request, url: %s', url)
        return requests.get(url, timeout=timeout, **kwargs)

    def ping(self):
        return self.get('/tvm/ping')

    def get_ticket(self, src, dst):
        resp = self.get(
            '/tvm/tickets',
            params={'src': src, 'dsts': dst},
            headers={'Authorization': self.AUTHTOKEN},
        )
        assert resp.status_code == 200
        tickets = list(resp.json().values())
        assert len(tickets) == 1
        return tickets[0]['ticket']

    def is_ticket_valid(self, tvm_ticket, dest_id):
        response = self.get(
            "/tvm/checksrv",
            params={
                'dst': dest_id,
            },
            headers={
                'Authorization': self.AUTHTOKEN,
                'X-Ya-Service-Ticket': tvm_ticket,
            },
        )
        if response.status_code != 200:
            return False
        data = json.loads(response.content)
        if int(data['dst']) != dest_id:
            return False
        return True
