import os

import requests
from werkzeug.wrappers import Response, Request
from werkzeug.wsgi import responder

from mail.devpack.lib import helpers
from mail.devpack.lib.components.base import BaseComponent, WithPort
from mail.devpack.lib.errors import DevpackError


class UnimockApp(object):
    def __init__(self, logger, response_body="ok"):
        self.logger = logger
        self.response_body = response_body

    @responder
    def __call__(self, environ, start_response):
        r = Request(environ)
        self.logger.info("[server] %s %s 200 '%s'", r.method, r.url, r.data)
        return Response(self.response_body)


class Unimock(BaseComponent, WithPort):
    NAME = "unimock"

    @classmethod
    def gen_config(cls, port_generator, config=None):
        base = super(Unimock, cls).gen_config(port_generator, config=config)
        return dict(
            tvm={"client_id": next(port_generator)},
            **base
        )

    def __init__(self, env, components):
        self.tvm_id = env.config['unimock']['tvm']['client_id']
        super(Unimock, self).__init__(env, components)
        self.app = UnimockApp(self.logger)

    def start(self):
        self.logger.info('starting unimock on port %d', self.port)
        child_pid = os.fork()
        if child_pid:
            try:
                helpers.wait_ping(self.logger, self.ping)
            except:
                raise DevpackError("unimock does not response to / after start")
            self.state["pid"] = child_pid
            self.logger.info('unimock started, pid: %d', child_pid)
        else:
            helpers.start_werkzeug(self)

    def stop(self):
        pid = self.state.get("pid")
        if pid:
            self.logger.info('stopping unimock on port %d with pid %d', self.port, pid)
            helpers.kill_proc(pid)
            del self.state["pid"]
            self.logger.info('unimock stopped')
        else:
            self.logger.warning('No pid in state for %s', self.name)

    def info(self):
        return {
            "state": self.state,
            "root": self.root,
            "port": self.port,
        }

    def ping(self):
        return self.get('/')

    def get(self, query, timeout=10):
        url = "http://localhost:{port}{query}".format(port=self.port, query=query)
        self.logger.info('GET request, url: ' + url)
        return requests.get(url, timeout=timeout)

    def post(self, query, data, timeout=10):
        url = "http://localhost:{port}{query}".format(port=self.port, query=query)
        self.logger.info('POST request, url: ' + url + ' data: ' + data)
        return requests.post(url, data=data, timeout=timeout)
