import os

import requests

from mail.devpack.lib import helpers
from mail.devpack.lib.components.base import BaseComponent, WithPort
from mail.devpack.lib.errors import DevpackError


class WerkzeugComponent(BaseComponent, WithPort):
    def __init__(self, env, components, app, is_https=False):
        super(WerkzeugComponent, self).__init__(env, components)
        self._is_https = is_https
        self._ssl_cert_path = None
        self.app = app

    @property
    def host(self):
        return "https://localhost" if self._is_https else "http://localhost"

    @property
    def ssl_cert_path(self):
        if self._is_https and not self._ssl_cert_path:
            # Important to do it after fork - nasty tempfile things are going on inside
            self._ssl_cert_path, _ = helpers.get_ssl_context()
        return self._ssl_cert_path

    @property
    def url(self):
        """No trailing slash"""
        return "{self.host}:{self.port}".format(self=self)

    def _request(self, query, timeout=10, data=None):
        url = "{self.url}{query}".format(self=self, query=query)
        self.logger.info('http call: %s', url)
        if data is not None:
            self.logger.info('http post data: %s', str(data))
            return requests.post(url, timeout=timeout, data=data, verify=self.ssl_cert_path)
        else:
            return requests.get(url, timeout=timeout, verify=self.ssl_cert_path)

    def start(self):
        self.logger.info('starting %s on port %d', self.NAME, self.port)
        child_pid = os.fork()
        if child_pid:
            try:
                helpers.wait_ping(self.logger, self.ping)
            except Exception as e:
                self.logger.exception(e)
                raise DevpackError("%s does not response to / after start" % self.NAME)
            self.state["pid"] = child_pid
            self.logger.info('%s started, pid: %d', self.NAME, child_pid)
        else:
            # TODO :: Config it
            helpers.start_werkzeug(self, https=self._is_https)

    def stop(self):
        pid = self.state.get("pid")
        if not pid:
            self.logger.info("%s is already not running", self.NAME)
            return
        self.logger.info('stopping %s on port %d with pid %d', self.NAME, self.port, pid)
        helpers.kill_proc(pid)
        del self.state["pid"]
        self.logger.info('%s stopped', self.NAME)

    def info(self):
        return {
            "state": self.state,
            "root": self.root,
            "port": self.port,
        }

    def ping(self):
        return self._request('/')
