import os
import yaml

from .errors import ConfigError
from . import components_manager

DEFAULT_CONFIG_PATH = os.path.expanduser('~/.devpack.yml')


def generate_port(get_port, offset=6000):
    for i in range(10000):
        port = i * 5 + offset
        yield get_port(port)


class Config(dict):
    def as_dict(self):
        return dict(self)

    @property
    def root(self):
        return self['sysdata']['root']

    def __getitem__(self, item):
        try:
            return super(Config, self).__getitem__(item)
        except KeyError:
            raise ConfigError('item %s not found in config, please update or re-generate it.' % item)


def read(path):
    if not os.path.isfile(path):
        raise ConfigError("config file %s not found, please generate it." % path)
    return Config(yaml.safe_load(open(path)))


def write(config, path):
    with open(path, 'w') as outfile:
        yaml.dump(config.as_dict(), outfile, default_flow_style=False)


def generate_config(get_port, root, top_comp_cls, arcadia=None, offset=6000):
    port_generator = generate_port(get_port, offset)
    res = {
        "arcadia": arcadia,
        "sysdata": {
            "root": root,
            "state_dir": "state",
        },
    }
    for comp in components_manager.toposorted(top_comp_cls=top_comp_cls):
        config = comp.gen_config(port_generator, config=res)
        if config:
            res.update({comp.NAME: config})
    return Config(res)


def generate_default_config(arcadia_root, offset, top_comp_cls):
    return generate_config(lambda port: port, "~/devpack", top_comp_cls, arcadia_root, offset)
