import logging
import os
import sys
from collections import namedtuple
from contextlib import contextmanager
from shutil import rmtree

from psutil import process_iter, AccessDenied, wait_procs, NoSuchProcess

from . import components_manager
from . import helpers
from . import state


class Coordinator(object):
    def __init__(self, env, top_comp_cls):
        self.top_comp_cls = top_comp_cls
        components_cls = components_manager.toposorted(self.top_comp_cls)
        self.config = env.get_config()
        self._init_state_dir()
        self.logger = self._init_logger(env)
        self.components = {}
        for comp_cls in components_cls:
            comp = comp_cls(env, self.components)
            self.components[comp_cls] = comp
        self._comps = namedtuple('Components', (c.NAME.replace('-', '_') for c in self.components.keys()))(
            *self.components.values()
        )
        """Easier access to components in interactive debugging"""

    def _init_state_dir(self):
        if self.config:
            wd = self.config['sysdata']['root']
            state_dir = os.path.join(wd, self.config['sysdata']['state_dir'])
            helpers.mkdir_recursive(state_dir)

    def _init_logger(self, env):
        logger = logging.getLogger(__name__)
        logger.setLevel(logging.DEBUG)
        if len(logger.handlers):
            return logger
        if env.log_stdout() and not len(logging.getLogger().handlers):
            stdout_handler = logging.StreamHandler(sys.stdout)
            stdout_formatter = logging.Formatter('[%(asctime)s] %(message)s')
            stdout_handler.setFormatter(stdout_formatter)
            logger.addHandler(stdout_handler)
        log_path = os.path.join(self.config["sysdata"]["root"], "coordinator.log")
        file_handler = logging.FileHandler(os.path.expanduser(log_path))
        file_formatter = logging.Formatter('[%(asctime)s] [%(process)d] %(message)s')
        file_handler.setFormatter(file_formatter)
        logger.addHandler(file_handler)
        return logger

    def _dump_state(self, component_cls):
        comp = self.components[component_cls]
        state.write_state(self.config, comp.name, comp.state)

    def apply_action_for_deps(self, action, component_cls, processed=None):
        processed = set() if processed is None else processed
        deps = self.components[component_cls].DEPS
        res = {}
        for dep in deps:
            if dep in processed:
                continue
            processed.add(dep)
            self.logger.info("%s: apply action for dependency %s", component_cls.NAME, dep.NAME)
            r = action(dep, with_deps=True, processed=processed)
            if r:
                res.update(r)
        return res

    def start(self, component_cls=None, with_deps=True, processed=None):
        if component_cls is None:
            component_cls = self.top_comp_cls
        if with_deps:
            self.apply_action_for_deps(self.start, component_cls, processed)
        comp = self.components[component_cls]
        try:
            if not comp.state.get('root_inited'):
                self.logger.info("%s: initializing root ...", component_cls.NAME)
                comp.init_root()
                comp.state['root_inited'] = True
            if not comp.state.get('started'):
                self.logger.info("%s: starting ...", component_cls.NAME)
                comp.start()
                comp.state['started'] = True
            if not comp.state.get('data_prepared'):
                self.logger.info("%s: preparing data ...", component_cls.NAME)
                comp.prepare_data()
                comp.state['data_prepared'] = True
        finally:
            self._dump_state(component_cls)
        self.logger.info("%s: successfully started", component_cls.NAME)

    def stop(self, component_cls=None, with_deps=True, processed=None):
        if component_cls is None:
            component_cls = self.top_comp_cls
        comp = self.components[component_cls]
        if comp.state.get('started'):
            self.logger.info("%s: stopping ...", component_cls.NAME)
            comp.stop()
            comp.state['started'] = False
            self.logger.info("%s: successfully stopped", component_cls.NAME)
        else:
            self.logger.info("%s: was not running", component_cls.NAME)
        self._dump_state(component_cls)
        if with_deps:
            self.apply_action_for_deps(self.stop, component_cls, processed)

    def hard_purge(self):
        def on_terminate(proc):
            self.logger.info("process %r terminated with exit code %d", proc, proc.returncode)

        wd = self.config['sysdata']['root']
        procs_to_kill = []
        for proc in process_iter():
            try:
                if proc.cwd().startswith(wd):
                    proc.terminate()
                    procs_to_kill.append(proc)
            except (AccessDenied, NoSuchProcess):
                pass
        _, alive = wait_procs(procs_to_kill, timeout=10, callback=on_terminate)
        for p in alive:
            p.kill()
        abs_wd = os.path.abspath(os.path.expanduser(wd))
        if os.path.exists(abs_wd):
            rmtree(abs_wd)

    def purge(self, component_cls=None, with_deps=True, processed=None):
        if component_cls is None:
            component_cls = self.top_comp_cls
        self.logger.info("purging %s ...", component_cls.NAME)
        self.stop(component_cls, with_deps=False)

        if with_deps:
            self.apply_action_for_deps(self.purge, component_cls)

        comp = self.components[component_cls]
        state_file = state.get_state_path(self.config, component_cls.NAME)
        if os.path.isfile(state_file):
            os.remove(state_file)

        if comp.is_multiroot():
            comp.purge()
        self.logger.info("%s: successfully purged", component_cls.NAME)

    def info(self, component_cls=None, with_deps=True, processed=None):
        if component_cls is None:
            component_cls = self.top_comp_cls
        res = {}
        if with_deps:
            res = self.apply_action_for_deps(self.info, component_cls)
        comp = self.components[component_cls]
        comp_info = comp.info()
        if comp_info:
            res[component_cls.NAME] = comp_info
        return res

    @contextmanager
    def context(self, component_cls=None, do_cleanup=True):
        if component_cls is None:
            component_cls = self.top_comp_cls
        try:
            self.start(component_cls)
            yield self
        finally:
            if do_cleanup:
                self.stop(component_cls)
