import errno
import logging
import os
import signal
import subprocess
import time
from os.path import isfile

import psutil
from six import binary_type
from werkzeug.serving import run_simple

import backoff
from .errors import DevpackError

import yatest.common


def write2file(what, path):
    if isinstance(what, binary_type):
        mode = "wb"
    else:
        mode = "w"
    with open(os.path.expanduser(path), mode) as f:
        f.write(what)


def read_line(path):
    with open(os.path.expanduser(path), "r") as f:
        return f.readline()


def create_logger(module_name, obj, log_file):
    logger = logging.getLogger("%s.%s" % (module_name, id(obj)))
    logger.setLevel(logging.DEBUG)
    formatter = logging.Formatter('[%(asctime)s] [%(levelname)s] [%(name)s] %(message)s')

    file_handler = logging.FileHandler(os.path.join(obj.root, log_file))
    file_handler.setFormatter(formatter)
    logger.addHandler(file_handler)

    # if not len(logging.getLogger().handlers):
    #     stdout_handler = logging.StreamHandler(sys.stdout)
    #     stdout_handler.setFormatter(formatter)
    #     logger.addHandler(stdout_handler)

    return logger


def create_root(name, working_dir=None, use_ram_disk_if_available=False):
    if not working_dir:
        working_dir = os.getcwd()
    root = os.path.join(working_dir, name)
    root = os.path.expanduser(root)
    if not os.path.isabs(root):
        if use_ram_disk_if_available and yatest.common.ram_drive_path():
            root = os.path.join(yatest.common.ram_drive_path(), 'testing_out_stuff', root)
        else:
            root = os.path.join('testing_out_stuff', root)
    root = os.path.abspath(root)
    if not os.path.exists(root):
        os.makedirs(root)
    return root


def kill_proc_the_right_way(pid, sig):
    try:
        os.kill(pid, sig)
    except OSError as err:
        if err.errno != errno.ESRCH:
            raise


def kill_proc(pid, sig=signal.SIGTERM):
    retries = 5
    for retry in range(0, retries):
        kill_proc_the_right_way(pid, sig)
        if not psutil.pid_exists(pid):
            return
        time.sleep(1)
    kill_proc_the_right_way(pid, signal.SIGKILL)


def kill_proc_by_pid_path(pid_path, sig=signal.SIGTERM):
    with open(pid_path) as f:
        kill_proc(pid=int(f.read()), sig=sig)


@backoff.on_exception(
    backoff.constant,
    Exception,
    max_tries=30,
    interval=2
)
def wait_ping(logger, get_ping, ping_text=None):
    ping = get_ping()
    logger.info("ping.status_code=%d, ping.text='%s'", ping.status_code, ping.text)
    if ping.status_code == 200 and (ping_text is None or ping.text == ping_text):
        logger.info('pong received, ok')
    else:
        raise DevpackError("pong does not match")


def mkdir_recursive(path):
    path = os.path.expanduser(path)
    if not os.path.exists(path):
        os.makedirs(path)


def get_ssl_pair():
    from library.python import resource as rs

    cert = list(rs.iteritems(prefix='resfs/file/mail/devpack/data/dev-cert.pem'))[0][1]
    pkey = list(rs.iteritems(prefix='resfs/file/mail/devpack/data/dev-key.pem'))[0][1]
    return cert, pkey


# Copy-pasted from werkzeug.serving.generate_adhoc_ssl_context
def get_ssl_context():
    """Generates an adhoc SSL context for the development server."""
    import tempfile
    import os

    cert, pkey = get_ssl_pair()
    cert_handle, cert_file = tempfile.mkstemp(suffix='.ca')
    pkey_handle, pkey_file = tempfile.mkstemp(suffix='.pkey')

    os.write(cert_handle, cert)
    os.write(pkey_handle, pkey)
    os.close(cert_handle)
    os.close(pkey_handle)
    return cert_file, pkey_file


def start_werkzeug(obj, https=False):
    file_handler = logging.FileHandler(os.path.join(obj.root, "werkzeug.log"))
    logging.getLogger('werkzeug').addHandler(file_handler)
    logging.getLogger('werkzeug').setLevel(logging.INFO)
    logging.getLogger('werkzeug').propagate = False
    run_simple(
        '::',
        obj.port,
        obj.app,
        use_debugger=True,
        use_reloader=False,
        ssl_context=get_ssl_context() if https else None,
    )


def run_subprocess(short_name, logger, cmd, env, cwd=None):
    try:
        output = subprocess.check_output(cmd, env=env, stderr=subprocess.STDOUT, cwd=cwd)
        logger.info("successfully executed '%s' done, output:\n%s", short_name, output)
    except subprocess.CalledProcessError as e:
        logger.info("failed to execute '%s', output:\n%s", short_name, e.output)
        raise
    return output.decode('utf-8')


def find_pid(root, name, args):
    for p in psutil.process_iter():
        try:
            if p.name() == name \
                    and p.cwd().startswith(os.path.realpath(root)) \
                    and set(p.cmdline()) & set(args):
                return p.pid
        except:
            pass
    return None


def wait_process(process, timeout):
    start = time.time()
    while True:
        process.poll()
        if process.returncode is not None:
            return True
        if time.time() - start > timeout:
            return False
        time.sleep(min(timeout / 10, 0.1))


def get_ubuntu_codename():
    if not isfile('/etc/lsb-release'):
        return None
    return next(
        (
            args[1] for args in map(
                lambda line: line.split('='),
                open('/etc/lsb-release').read().split()
            )
            if args[0] == 'DISTRIB_CODENAME'),
        None
    )
