import os
import os.path
import subprocess
import requests
import string
from six import binary_type

from . import helpers
from .errors import DevpackError
from library.python import resource


class YplatformAppHttpService(object):
    @staticmethod
    def gen_config(port_generator, config=None):
        return {
            "webserver_port": next(port_generator),
            "statserver_port": next(port_generator),
        }

    def __init__(self, env, name, arcadia_path='mail', custom_path=None):
        self.config = env.get_config()
        self.name = name

        self.webserver_port = self.config[self.name]['webserver_port']
        self.statserver_port = self.config[self.name]['statserver_port']

        self.bin_file = env.get_arcadia_bin('{path}/{name}/bin/application'.format(path=arcadia_path, name=custom_path or self.name))

        working_dir = self.config['sysdata']['root']

        self.root = helpers.create_root("%s_%d" % (self.name, self.webserver_port), working_dir)
        self.logger = helpers.create_logger(__name__, self, "%s-wrapper.log" % self.name)

        self.env = dict(os.environ)

    def get_app_path(self):
        return os.path.join(self.root, 'app')

    def get_config_path(self):
        return os.path.join(self.get_app_path(), 'config')

    def get_secrets_path(self):
        return os.path.join(self.get_app_path(), 'secrets')

    def get_app_config_path(self):
        return os.path.join(self.get_app_path(), 'config', 'config-devpack.yml')

    def get_pid_path(self):
        return os.path.join(self.root, 'app', 'application.pid')

    def get_resources_path(self):
        return os.path.join(self.get_app_path(), 'resources')

    def init_root(self):
        helpers.mkdir_recursive(self.get_config_path())
        helpers.mkdir_recursive(self.get_secrets_path())
        helpers.mkdir_recursive(os.path.join(self.get_app_path(), 'log'))
        helpers.mkdir_recursive(self.get_resources_path())

    @staticmethod
    def init_pgpass(root):
        pgpassfile = os.path.join(root, 'app', 'config', 'pgpass')
        helpers.write2file(resource.find('macs_pg/pgpass'), pgpassfile)
        os.chmod(pgpassfile, 0o600)

    def get_root(self):
        return self.root

    def format_config(self, config, **kwargs):
        class SafeDict(dict):
            def __missing__(self, key):
                return '{' + key + '}'

        if isinstance(config, binary_type):
            config = config.decode('utf-8')

        return string.Formatter().vformat(config, (), SafeDict(
            dir=self.root,
            pid=self.get_pid_path(),
            webserver_port=self.webserver_port,
            statserver_port=self.statserver_port,
            **kwargs
        ))

    def start(self, ping_response):
        self.logger.info("starting %s on port %d", self.name, self.webserver_port)
        config_path = self.get_app_config_path()
        if not os.path.exists(self.bin_file):
            raise DevpackError("%s binary does not exist at %s" % (self.name, self.bin_file))
        if not os.path.exists(config_path):
            raise DevpackError("%s config does not exist at %s" % (self.name, config_path))
        cmd = [self.bin_file, config_path]
        try:
            environ = getattr(self, 'env', None)
            self.logger.info("executing command %s", ' '.join('"%s"' % v for v in cmd))
            process = subprocess.Popen(cmd, cwd=self.root, stdout=subprocess.PIPE,
                                       stderr=subprocess.PIPE, env=environ)
            if not helpers.wait_process(process=process, timeout=10):
                self.logger.info("%s service did not started within timeout, output:\n%s\n%s",
                                 self.name, process.stdout.read(), process.stderr.read())
                process.kill()
                raise DevpackError("%s did not started within timeout" % self.name)
            if process.returncode != 0:
                self.logger.info("%s service failed to daemonize, output:\n%s\n%s", self.name,
                                 process.stdout.read(), process.stderr.read())
                raise DevpackError("%s service failed to daemonize" % self.name)
            self.logger.info("%s started output:\n%s\n%s", self.name,
                             process.stdout.read(), process.stderr.read())
        except subprocess.CalledProcessError as e:
            self.logger.info("%s start failed, output:\n%s", self.name, e.output)
            raise
        try:
            if ping_response is not None:
                helpers.wait_ping(self.logger, self.ping, ping_response)
        except:
            self.stop()
            raise DevpackError("%s does not response to /ping after start" % self.name)

    def stop(self):
        self.logger.info("stopping %s on port %d", self.name, self.webserver_port)
        if not os.path.exists(self.get_pid_path()):
            self.logger.info("%s is not running", self.name)
            return
        helpers.kill_proc_by_pid_path(self.get_pid_path())
        self.logger.info("%s stopped", self.name)

    def info(self):
        return {
            "root": self.root,
            "bin_file": self.bin_file,
            "webserver_port": self.webserver_port,
            "statserver_port": self.statserver_port,
        }

    @property
    def url(self):
        return "http://[::1]:{port}".format(port=self.webserver_port)

    def get(self, target, timeout=10, params=None, headers={}):
        url = "{url}{target}".format(url=self.url, target=target)
        self.logger.info('GET request, target: %s, params: %s', url, params)
        return requests.get(url, timeout=timeout, params=params, headers=headers)

    def post(self, target, data, timeout=10, headers={}):
        url = "{url}{target}".format(url=self.url, target=target)
        self.logger.info('POST request, url: %s, data: %s', url, data)
        return requests.post(url, data=data, timeout=timeout, headers=headers)

    def ping(self):
        return self.get('/ping')

    @staticmethod
    def make_url(endpoint, **kwargs):
        args = []
        for key, value in kwargs.items():
            if value is not None:
                args.append('{k}={v}'.format(k=key, v=value))
        if len(args) == 0:
            return '/{}'.format(endpoint)
        return '/{}?{}'.format(endpoint, '&'.join(args))

    def update_env(self, values):
        self.env.update(values)

    def put_pgpassfile_in_env(self, root):
        self.update_env({'PGPASSFILE': os.path.join(self.get_app_path(), 'config', 'pgpass')})

    def tweak_env_for_asan(self):
        asan_options = ''
        if 'ASAN_OPTIONS' in self.env:
            asan_options = self.env['ASAN_OPTIONS'] + ':'
        assert 'detect_stack_use_after_return' not in asan_options
        asan_options += 'detect_stack_use_after_return=1'
        self.update_env({'ASAN_OPTIONS': asan_options})
