import os
import shutil
import socket
import backoff
import json
import yaml

from library.python import resource
from mail.devpack.lib import helpers
from mail.devpack.lib.state import read_state
from mail.devpack.lib.errors import DevpackError
from mail.devpack.lib.components.base import SubprocessComponent, AbstractComponent
from mail.devpack.lib.components.sharpei import Sharpei
from mail.devpack.lib.components.mdb import Mdb
from mail.webmail_config.lib.make_config import make_config


SERVICE_YAML_TEMPLATE = '''
devpack:
    sharpei:
        scheme: http
        host: localhost
        port: %(sharpei_port)s
    database:
        user: doberman
        query_conf: %(query_conf)s
    change_queue:
        pull_tries: 5
    control:
        port: %(port)s
    log_dir: %(log_dir)s
'''


class DobermanInstance(SubprocessComponent):
    NAME = "doberman-instance"
    DEPS = [Sharpei, Mdb]

    @classmethod
    def gen_config(cls, port_generator, config=None):
        base = super(DobermanInstance, cls).gen_config(port_generator, config=config)
        return dict(
            port=next(port_generator),
            **base
        )

    def __init__(self, env, components, instance, instance_conf):
        self.instance = instance
        self.__port = instance_conf['port']
        super(DobermanInstance, self).__init__(env, components)
        self.sharpei = components[Sharpei]

        self.bin_file = env.get_arcadia_bin('mail/doberman/src/doberman')
        self.config_path = os.path.join(self.etc_path, 'doberman.conf-devpack')
        self.etc_macs_pg_path = os.path.join(self.root, 'etc', 'macs_pg')
        self.query_conf_path = os.path.join(self.etc_macs_pg_path, 'query.conf')

    @property
    def root_name(self):
        return "%s_%s" % (self.name, self.instance)

    @property
    def port(self):
        return self.__port

    def __copy_query_conf(self):
        helpers.mkdir_recursive(self.etc_macs_pg_path)
        helpers.write2file(resource.find("macs_pg/query.conf"), self.query_conf_path)

    def __dump_service_cfg(self):
        text = SERVICE_YAML_TEMPLATE % {
            'sharpei_port': self.sharpei.webserver_port(),
            'query_conf': self.query_conf_path,
            'port': self.port,
            'log_dir': self.log_dir,
        }
        return yaml.safe_load(text)

    def __generate_config(self, service_cfg):
        base = resource.find('doberman/doberman.conf')
        config = make_config('devpack', base, service_cfg, silent=True) % {
            'sharpei_port': self.sharpei.webserver_port(),
            'query_conf': self.query_conf_path,
            'port': self.port,
            'log_dir': self.log_dir,
        }
        helpers.write2file(config, os.path.join(self.etc_path, 'doberman.conf-devpack'))

    def init_root(self):
        helpers.mkdir_recursive(self.etc_path)
        helpers.mkdir_recursive(self.log_dir)
        self.__copy_query_conf()
        self.__generate_config(self.__dump_service_cfg())

    def __ping(self):
        return self.__get_stat() is not None

    @backoff.on_exception(
        backoff.constant,
        Exception,
        max_tries=30,
        interval=1
    )
    def __wait_ping(self):
        if self.__ping():
            self.logger.info('stat received, ok')
        else:
            raise DevpackError("stat has not been received")

    def __find_pid(self):
        return helpers.find_pid(self.root, 'doberman', [self.config_path])

    def __command(self, command):
        try:
            sock = socket.create_connection(("localhost", self.port), 5)
            sock.sendall(bytearray(command + '\n', encoding='utf-8'))
            response = sock.recv(65536)
            sock.close()
            try:
                res = json.loads(response)
            except ValueError as e:
                self.logger.warning('Not JSON object', exc_info=e)
                res = response
        except Exception as e:
            self.logger.exception(e)
            res = None
        return res

    def __get_stat(self):
        return self.__command('stat')

    def start(self):
        pid = self.__find_pid()
        if pid is None:
            cmd = [self.bin_file, self.config_path]
            with self.starting(cmd):
                self.__wait_ping()

    def stop(self):
        pid = self.__find_pid()
        if pid is not None:
            helpers.kill_proc(pid)

    def purge(self):
        if os.path.exists(self.root):
            shutil.rmtree(self.root, ignore_errors=True)

    def info(self):
        pid = self.__find_pid()
        self.state['started'] = pid is not None
        return {
            "root": self.root,
            "bin_file": self.bin_file,
            "port": self.port,
            "state": self.state,
            "pid": pid,
            "info": self.__get_stat(),
        }

    def ping(self):
        pid = self.__find_pid()
        if pid is not None:
            return self.__ping()
        return False

    def command(self, command):
        return self.__command(command)


def dobby_name(idx):
    return 'Dobby#%d' % (idx + 1)


class Doberman(AbstractComponent):
    NAME = "doberman"
    DEPS = [Sharpei, Mdb]

    @staticmethod
    def gen_config(port_generator, config):
        return {
            dobby_name(i): {"port": next(port_generator)}
            for i in range(len(config[Mdb.NAME]['shards']))
        }

    def __init__(self, env, components):
        self._dobbys = {
            dobby_name: DobermanInstance(env, components, instance=dobby_name, instance_conf=inst_conf)
            for dobby_name, inst_conf in env.get_config()[self.NAME].items()
        }
        self.__state = read_state(env.get_config(), self.NAME)

    @property
    def state(self):
        return self.__state

    def info(self):
        return {
            name: inst.info()
            for name, inst in self._dobbys.items()
        }

    def init_root(self):
        for dobby in self._dobbys.values():
            dobby.init_root()

    def start(self):
        for dobby in self._dobbys.values():
            dobby.start()

    def stop(self):
        for dobby in self._dobbys.values():
            dobby.stop()

    def purge(self):
        for dobby in self._dobbys.values():
            dobby.purge()

    def ping(self):
        return all(dobby.ping() for dobby in self._dobbys.values())

    def dobby(self, idx=None):
        idx = idx or 0
        if isinstance(idx, int):
            idx = dobby_name(idx)
        return self._dobbys.get(idx)

    def is_multiroot(self):
        return True
