#ifndef DOBERMAN_SRC_CORO_SPAWN_H_
#define DOBERMAN_SRC_CORO_SPAWN_H_

#include <boost/asio/spawn.hpp>
#include <src/access/factory.h>
#include <src/logic/agent_launcher.h>

namespace doberman {
namespace access {

template <typename T, typename Log, typename Profiler>
inline auto makeAgentSpawner(boost::asio::io_service& ios, AccessFactory<T> makeAccess,
                             Log log, Profiler) {
    return [&ios, makeAccess = std::move(makeAccess), log = std::move(log)] (auto work) {
        boost::asio::spawn(ios,
                [&ios, work = std::move(work), makeAccess, log]
                (boost::asio::yield_context yield) mutable {
            try {
                work(makeAccess(yield));
            } catch (const boost::coroutines::detail::forced_unwind&) {
                throw;
            } catch (const WorkerIdOutdated&) {
                DOBBY_LOG_ERROR(log, "AgentSpawner: worker id is outdated, no work can be performed, exit immediately");
                ios.stop();
            } catch (const std::exception& e) {
                DOBBY_LOG_ERROR(log, "AgentSpawner: unexpected exception", log::exception=e);
            } catch (...) {
                DOBBY_LOG_ERROR(log, "AgentSpawner: unexpected unknown exception");
            }
        });
    };
}

template <typename T, typename Log, typename Profiler>
inline auto spawn(
                boost::asio::io_service& ios,
                int maxSubscriptionsCount,
                AccessFactory<T> makeAccess,
                Log log,
                Profiler profiler,
                LabelFilter labelFilter,
                const service_control::RunStatus& runStatus) {

    boost::asio::spawn(ios, [&ios, makeAccess = std::move(makeAccess),
                             log = std::move(log), profiler = std::move(profiler),
                             runStatus = &runStatus,
                             labelFilter = std::move(labelFilter),
                             maxSubscriptionsCount]
                             (boost::asio::yield_context yield) mutable {
        try {
            auto repo = makeAccess(yield).subscriptionRepository();
            auto launcher = ::doberman::logic::makeAgentLauncher(
                    ::doberman::logic::makeSubscriptionRepository(std::move(repo), maxSubscriptionsCount),
                    makeAgentSpawner(ios, std::move(makeAccess), log, profiler),
                    log,
                    profiler,
                    std::move(labelFilter),
                    *runStatus);
            launcher.run();
            return;
        } catch (const boost::coroutines::detail::forced_unwind&) {
            throw;
        } catch (const WorkerIdOutdated&) {
            DOBBY_LOG_ERROR(log, "launcher: worker id is outdated, no work can be performed, exit immediately");
        } catch (const std::exception& e) {
            DOBBY_LOG_ERROR(log, "launcher: unexpected exception, exit immediately", log::exception=e);
        } catch (...) {
            DOBBY_LOG_ERROR(log, "launcher: unexpected unknown exception, exit immediately");
        }
        ios.stop();
    });
}

} // namespace access
} // namespace doberman

#endif /* DOBERMAN_SRC_CORO_SPAWN_H_ */
