#ifndef DOBERMAN_SRC_CORO_SUBSCRIPTION_REPOSITORY_ACCESS_H_
#define DOBERMAN_SRC_CORO_SUBSCRIPTION_REPOSITORY_ACCESS_H_

#include <boost/asio/spawn.hpp>
#include <src/logic/subscription_repository.h>

namespace doberman {
namespace access {

template <typename Impl>
class SubscriptionRepository {
    using Yield = boost::asio::yield_context;
    using ImplCtx = doberman::logic::AccessContext<Impl>;
    using ReleaseList = std::vector<std::function<void(ImplCtx&, const Impl&, Yield)>>;
public:
    SubscriptionRepository(Impl impl, boost::asio::yield_context yield)
    : impl(impl), yield(yield) {}

    template <typename ... Args>
    auto makeContext(Args&& ... args) const {
        return std::make_tuple(
                detail::dereference(impl).makeContext(std::forward<Args>(args)...),
                ReleaseList{});
    }

    template <typename Ctx>
    auto getReserved(Ctx& ctx) const {
        return detail::dereference(impl).getReserved(original(ctx), yield);
    }

    template <typename Ctx>
    auto reserve(Ctx& ctx, int credit) const {
        applyReleaseList(ctx);
        return detail::dereference(impl).reserve(original(ctx), credit, yield);
    }

    template <typename Ctx>
    void release(Ctx& ctx, SubscriptionId id) const {
        /**
         * Since release() may be called from different coroutine we can not
         * use the yield context here. Instead - we place the id into list, which
         * will be handled at reserve() method right before implementation of
         * reserve is called. So we can remove this hack after transfer from
         * naked boost::asio::spawn to boost::fibers.
         */
        releaseList(ctx).push_back(
                [id] (ImplCtx& ctx, const Impl& impl, Yield yield) {
            detail::dereference(impl).release(ctx, id, yield);
        });
    }

    template <typename Ctx>
    void decline(Ctx& ctx, SubscriptionId id, std::string reason) const {
        /**
         * Since decline() may be called from different coroutine we can not
         * use the yield context here. Instead - we place the id into list, which
         * will be handled at reserve() method right before implementation of
         * reserve is called. So we can remove this hack after transfer from
         * naked boost::asio::spawn to boost::fibers.
         */
        releaseList(ctx).push_back(
                [id, reason] (ImplCtx& ctx, const Impl& impl, Yield yield) {
            detail::dereference(impl).decline(ctx, id, reason, yield);
        });
    }
private:
    template <typename Ctx>
    static auto& releaseList(Ctx& ctx) { return std::get<ReleaseList>(ctx);}

    template <typename Ctx>
    void applyReleaseList(Ctx& ctx) const {
        auto& rlist = releaseList(ctx);
        while (!rlist.empty()) {
            auto f = std::move(rlist.back());
            rlist.pop_back();
            f(original(ctx), impl, yield);
        }
    }

    template <typename Ctx>
    static auto& original(Ctx& ctx) { return std::get<0>(ctx);}

    Impl impl;
    boost::asio::yield_context yield;
};

template <typename Impl>
inline auto makeSubscriptionRepository(Impl impl, boost::asio::yield_context yield) {
    return SubscriptionRepository<Impl>(std::move(impl), std::move(yield));
}

} // namespace access
} // namespace doberman

#endif /* DOBERMAN_SRC_CORO_SUBSCRIPTION_REPOSITORY_ACCESS_H_ */
