#pragma once

#include <macs_pg/changelog/change.h>

#include <src/access_impl/wrap_yield.h>
#include <src/access_impl/timer.h>
#include <src/access_impl/change_log_cache.h>

#include <src/detail/dereference.h>
#include <src/meta/types.h>
#include <src/profiling/profiler.h>
#include <src/logger/logger.h>
#include <coroutine_mutex/coroutine_mutex.hpp>
#include <boost/range/adaptors.hpp>
#include <boost/range/algorithm_ext/erase.hpp>
#include <boost/range/algorithm_ext/push_back.hpp>

BOOST_FUSION_DEFINE_STRUCT((doberman)(access_impl), ChangeCacheTimes,
        (std::uint64_t, flush_removed_changes_seconds)
)

namespace doberman {
namespace access_impl {

template <typename Shard,
          typename ChangeLogCache,
          typename Profiler,
          typename WorkerId,
          typename Log,
          typename Mutex = ::coro::Mutex,
          typename Clock = std::chrono::steady_clock>
class ChangeCache : public std::enable_shared_from_this<
        ChangeCache<Shard, ChangeLogCache, Profiler, WorkerId, Log, Mutex, Clock> > {
public:
    using Yield = typename Mutex::YieldContext;
    using Duration = typename Clock::duration;
    using Change = logic::Change;
    using ChangePtr = std::shared_ptr<const Change>;

    ChangeCache(WorkerId workerId, Shard shard, Duration ttl, Duration flushTimeoutSeconds,
                ChangeLogCache changeLogCache, Profiler profiler, Log log)
            : workerId_(std::move(workerId)), shard_(std::move(shard)), ttl_(ttl),
              flushTimeoutSeconds_(flushTimeoutSeconds), changeLogCache_(std::move(changeLogCache)),
              profiler_(profiler), log_(log) {}

    ChangePtr get(Uid uid, ::macs::SubscriptionId sid, Yield yield) {
        throwIfOutdated(workerId_);

        auto change = top(uid, sid);
        if (!change && expired(now())) {
            ++fetchWaitCount_;
            ::coro::unique_lock<Mutex> l(fetchMutex_, yield, 0);
            if (expired(now())) {
                fetch(yield);
            }
            l.unlock();
            change = top(uid, sid);
        }
        return change ? change->get(yield) : nullptr;
    }

    void flush(Yield yield) {
        if (!trash_.empty()) {
            ::coro::unique_lock<Mutex> l(fetchMutex_, yield, 0);
            flushTrashCache(yield);
        }
    }

    void remove(Uid uid, ::macs::SubscriptionId sid, ChangeId cid, Yield yield) {
        throwIfOutdated(workerId_);

        trash_.emplace_back(cid, uid, sid);
        pop(uid, sid, cid);

        if (timerScheduled_) {
            return;
        }
        using boost::asio::spawn;
        spawn(yield, [timeout = flushTimeoutSeconds_, selfPtr = std::weak_ptr<ChangeCache>(this->shared_from_this())]
                (auto yield) {
            timer::wait(timeout, yield);
            if (auto self = selfPtr.lock()) {
                try {
                    self->flush(yield);
                } catch (const std::exception& e) {
                    DOBBY_LOG_ERROR_WHERE(::doberman::detail::dereference(self->log_), "unexpected exception: ",
                            log::exception=e);
                }
                self->timerScheduled_ = false;
            }
        });
        timerScheduled_ = true;
    }

private:
    using Timepoint = typename Clock::time_point;
    using Sid = ::macs::SubscriptionId;
    using Key = std::tuple<Uid, Sid>;
    using ChangeProxyPtr = decltype(
            ::doberman::detail::dereference(std::declval<ChangeLogCache>()).getChange({}));
    using Queue = std::map<ChangeId, ChangeProxyPtr>;
    using Cache = std::map<Key, Queue>;
    using EmptyQueues = std::vector<typename Cache::iterator>;
    using Trash = std::vector<::macs::ChangeReference>;

    bool expired(Timepoint now) const {
        return ttl_ <= now - lastUpdateTime_;
    }

    static auto now() { return Clock::now(); }

    ChangeProxyPtr top(const Uid& uid, const Sid& sid) const {
        const auto i = cache_.find(std::tie(uid, sid));
        if (i==cache_.end() || i->second.empty()) {
            return nullptr;
        }
        return i->second.begin()->second;
    }

    Queue& queue(const Uid& uid, const Sid& sid) {
        const auto key = std::tie(uid, sid);
        auto i = cache_.find(key);
        if (i == cache_.end()) {
            std::tie(i, std::ignore) = cache_.emplace(key, Queue{});
        }
        return i->second;
    }

    void pop(const Uid& uid, const Sid& sid, const ChangeId& cid) {
        const auto i = cache_.find(std::tie(uid, sid));
        if (i!=cache_.end()) {
            i->second.erase(cid);
            if (i->second.empty()) {
                emptyQueues_.push_back(i);
            }
        }
    }

    void removeEmptyQueues() {
        using namespace ::boost;

        const auto busy = [](auto i) {return !i->second.empty();};
        remove_erase_if(emptyQueues_, busy);

        const auto less = [](auto& l, auto&r) {return l->first < r->first;};
        for_each(unique(sort(emptyQueues_, less)), [&](auto i) {this->cache_.erase(i);});

        emptyQueues_.clear();
    }

    void flushTrashCache(Yield yield) {
        throwIfOutdated(workerId_);
        if (!trash_.empty()) {
            repo().remove(trash_, wrap(yield));
            trash_.clear();
        }
    }

    void fetch(Yield yield) {
        const auto start = profiler().now();
        using namespace ::boost;
        using adaptors::transformed;

        const auto resolveQueue = [&](const ::macs::ChangeReference& ref) {
            auto& q = this->queue(ref.uid, ref.subscriptionId);
            return std::make_tuple(std::addressof(q), ref.change);
        };

        using Item = decltype(resolveQueue({}));
        const auto queue = [](const Item& item) -> Queue& {return *std::get<Queue*>(item);};
        const auto id = [](const Item& item) {return std::get<ChangeId>(item);};

        flushTrashCache(yield);

        std::vector<Item> items;
        push_back(items, repo().get(workerId_, wrap(yield)) | transformed(resolveQueue));

        const auto fetched = [&](const Item& item) {return queue(item).count(id(item));};
        remove_erase_if(items, fetched);

        std::vector<ChangeId> ids;
        ids.reserve(items.size());
        push_back(ids, items | transformed(id));
        changeRepo().update(unique(sort(ids)), yield);

        const auto enqueueChange = [&](const Item& item) {
            queue(item).emplace(id(item), this->getChange(id(item)));
        };
        for_each(items, enqueueChange);

        removeEmptyQueues();

        profiler().write("change_cache", "fetch", std::to_string(fetchWaitCount_), profiler().passed(start));
        lastUpdateTime_ = now();
        fetchWaitCount_ = 0;
    }

    auto getChange(ChangeId id) const {
        if (auto change = changeRepo().getChange(id)) {
            return change;
        }
        /*
        There are 2 cases, when we can't get by change_id:
            1. We got `old` change - this change_log part was truncated
            2. We got `fresh` change_id and read change_log from replica:

        Solution:
            1. wait for - [MAILPG-1270]
            2. always read from master
        */
        throw std::logic_error( std::string(__PRETTY_FUNCTION__) +
            " can't get changelog entry for \"" +
            std::to_string(id) + "\"");
    }

    auto& repo() const {
        return ::doberman::detail::dereference(shard_).changeQueue();
    }

    auto& changeRepo() {
        return ::doberman::detail::dereference(changeLogCache_);
    }

    const auto& changeRepo() const {
        return ::doberman::detail::dereference(changeLogCache_);
    }

    auto& profiler() const {
        return ::doberman::detail::dereference(profiler_);
    }

    WorkerId workerId_;
    Shard shard_;
    Cache cache_;
    Mutex fetchMutex_;
    Duration ttl_;
    Duration flushTimeoutSeconds_;
    Timepoint lastUpdateTime_ = now() - ttl_;
    ChangeLogCache changeLogCache_;
    EmptyQueues emptyQueues_;
    unsigned fetchWaitCount_ = 0;
    Profiler profiler_;
    Log log_;
    Trash trash_;
    bool timerScheduled_ = false;
};

} // namespace access_impl
} // namespace doberman

