#ifndef DOBERMAN_SRC_ACCESS_IMPL_CHANGE_COMPOSER_H_
#define DOBERMAN_SRC_ACCESS_IMPL_CHANGE_COMPOSER_H_

#include <algorithm>

#include <macs_pg/changelog/change.h>

#include <src/access_impl/wrap_yield.h>
#include <src/access_impl/timer.h>

#include <src/detail/dereference.h>
#include <src/logic/change.h>
#include <src/logic/types.h>
#include <src/meta/types.h>
#include <src/meta/changed.h>
#include <coroutine_mutex/coroutine_mutex.hpp>

namespace doberman {
namespace access_impl {

namespace compose {

using SubscribedFolder = ::doberman::logic::Change::SubscribedFolder;
using meta::labels::LabelsCache;

template <typename Profiler>
struct PutMessages {
    std::vector<EnvelopeWithMimes> envelopes;
    LabelsCache labels;
    Profiler profiler;

    ::macs::error_code operator()(const SubscribedFolder& to) const {
        const auto start = profiler.now();
        for (auto e: envelopes) {
            to.put(std::move(e), labels);
        }
        profiler.write("change_apply", "put", profiler.passed(start));
        return ::macs::error_code();
    }
};

template <typename Profiler>
struct MoveMessages {
    std::vector<EnvelopeWithMimes> envelopes;
    LabelsCache labels;
    Fid fid;
    Profiler profiler;

    ::macs::error_code move(const SubscribedFolder& to) const {;
        const auto start = profiler.now();
        for (auto e: envelopes) {
            to.put(std::move(e), labels);
        }
        profiler.write("change_apply", "move", profiler.passed(start));
        return ::macs::error_code();
    }

    ::macs::error_code remove(const SubscribedFolder& to) const {
        const auto start = profiler.now();
        auto mids = envelopes | boost::adaptors::transformed([](auto& e){
            return std::get<Envelope>(e).mid();
        });
        to.erase({std::begin(mids), std::end(mids)});
        profiler.write("change_apply", "remove", profiler.passed(start));
        return ::macs::error_code();
    }

    ::macs::error_code operator()(const SubscribedFolder& to) const {
        return fid == to.fid() ? move(to) : remove(to);
    }
};

template <typename Profiler>
struct EraseMessages {
    ::macs::MidList mids;
    Profiler profiler;

    ::macs::error_code operator()(const SubscribedFolder& to) const {
        const auto start = profiler.now();
        to.erase({std::begin(mids), std::end(mids)});
        profiler.write("change_apply", "erase", profiler.passed(start));
        return ::macs::error_code();
    }
};

template <typename Profiler>
struct UpdateMessages {
    ::macs::MidList mids;
    std::vector<Label> mark;
    std::vector<Label> unmark;
    Profiler profiler;

    ::macs::error_code operator()(const SubscribedFolder& to) const {
        const auto start = profiler.now();
        for (const auto& mid: mids) {
            if(!unmark.empty()) {
                to.unmark(mid, unmark);
            }
            if(!mark.empty()) {
                to.mark(mid, mark);
            }
        }
        profiler.write("change_apply", "update", profiler.passed(start));
        return ::macs::error_code();
    }
};

template <typename Profiler>
struct JoinThreads {
    ThreadId tid;
    std::vector<ThreadId> joinTids;
    Profiler profiler;

    ::macs::error_code operator()(const SubscribedFolder& to) const {
        const auto start = profiler.now();
        to.joinThreads(std::move(tid), std::move(joinTids));
        profiler.write("change_apply", "join threads", profiler.passed(start));
        return ::macs::error_code();
    }
};

} // namespace compose


template <typename MailboxProvider, typename Profiler>
class ChangeComposer {
    MailboxProvider mailboxProvider_;
    Profiler profiler_;
public:

    ChangeComposer(MailboxProvider getMailbox, Profiler profiler)
    : mailboxProvider_(getMailbox), profiler_(profiler) {}

    using Change = ::doberman::logic::Change;
    using ChangePtr = std::shared_ptr<const Change>;
    template <typename Yield>
    ChangePtr operator()(const ::macs::Change& changeLogEntry, Yield yield) const {
        return std::make_shared<Change>(
            changeLogEntry.changeId(),
            changeLogEntry.revision(),
            std::move(compose(changeLogEntry, yield))
        );
    }

private:
    using Action = ::doberman::meta::changed::ChangeAction;
    using Handler = ::doberman::logic::Change::Impl;
    using MoveArgs = ::doberman::meta::changed::MoveArguments;

    template <typename Yield>
    Handler compose(const ::macs::Change& changeLogEntry, Yield yield) const {
        const auto action = meta::changed::getActionByChange(changeLogEntry);
        using meta::changed::getMidsFromChange;
        switch (action) {
            case Action::put:
                return put(getMidsFromChange(changeLogEntry), changeLogEntry, std::move(yield));
            case Action::move:
                return move(getMidsFromChange(changeLogEntry), changeLogEntry, std::move(yield));
            case Action::erase:
                return compose::EraseMessages<Profiler>{getMidsFromChange(changeLogEntry), profiler_};
            case Action::update:
                return update(getMidsFromChange(changeLogEntry), changeLogEntry, std::move(yield));
            case Action::joinThreads:
                return joinThreads(changeLogEntry);
        }
        throw std::logic_error(
            std::string(__PRETTY_FUNCTION__) +
            " got unexpected action: '" + static_cast<char>(action) +
            "' while processing change.id: " + std::to_string(changeLogEntry.changeId())
        );
    }

    template <typename Yield>
    auto put(::macs::MidList mids, const ::macs::Change& changeLogEntry, Yield yield) const {
        auto mailbox = getMailbox(changeLogEntry.uid());
        auto labels = getAllLabels(*mailbox, yield);
        auto envMimes = getEnvelopesByMids(*mailbox, std::move(mids), yield);

        return compose::PutMessages<Profiler>{
            envMimes, std::move(labels), profiler_};
    }

    template <typename Yield>
    auto move(::macs::MidList mids, const ::macs::Change& changeLogEntry, Yield yield) const {
        auto mailbox = getMailbox(changeLogEntry.uid());
        auto labels = getAllLabels(*mailbox, yield);
        auto envMimes = getEnvelopesByMids(*mailbox, std::move(mids), yield);
        const auto args = meta::changed::getArguments<MoveArgs>(changeLogEntry);

        return compose::MoveMessages<Profiler>{
            envMimes, std::move(labels), std::to_string(args.fid), profiler_};
    }

    template <typename Yield>
    auto update(::macs::MidList mids, const ::macs::Change& changeLogEntry, Yield yield) const {
        auto mailbox = getMailbox(changeLogEntry.uid());
        auto labels = getAllLabels(*mailbox, yield);
        compose::UpdateMessages<Profiler> retval{std::move(mids), {}, {}, profiler_};

        std::vector<::macs::Lid> notFound;
        std::tie(retval.mark, retval.unmark, notFound) = meta::changed::getUpdatedLabels(
                    changeLogEntry, *labels);
        if (!notFound.empty()) {
            std::tie(retval.mark, retval.unmark, std::ignore) = meta::changed::getUpdatedLabels(
                        changeLogEntry, *(labels.update()));
        }
        return std::move(retval);
    }

    auto joinThreads(const ::macs::Change& changeLogEntry) const {
        compose::JoinThreads<Profiler> retval{{}, {}, profiler_};
        std::tie(retval.tid, retval.joinTids) = meta::changed::getJoinedThreads(changeLogEntry);
        return retval;
    }

    template <typename Mailbox, typename Yield>
    auto getAllLabels(Mailbox& m, Yield yield) const {
        auto& repo = m.labels();
        const auto labels = repo.getAllLabels(wrap(yield));
        return meta::labels::LabelsCache(labels, [&repo, yield]{
            repo.resetLabelsCache();
            return repo.getAllLabels(wrap(yield));
        });
    }

    template <typename Mailbox, typename Yield>
    std::vector<EnvelopeWithMimes> getEnvelopesByMids(
            Mailbox& m, macs::MidList mids, Yield yield) const {
        const auto& query = m.envelopes().query().withMimes().byMids(mids);
        const auto getEnvelopes = [&query](auto yield) {
            auto envelopes = query.get(yield);
            return std::vector<EnvelopeWithMimes>{envelopes.begin(), envelopes.end()};
        };

        try {
            return getEnvelopes(wrap(yield));
        } catch (const system_error& e) {
            if (e.code() != macs::error::noSuchLabel) {
                throw;
            }
        }
        m.labels().resetLabelsCache();
        return getEnvelopes(wrap(yield));
    }

    auto getMailbox(const Uid& uid) const {
        return ::doberman::detail::dereference(mailboxProvider_)(uid);
    }
};

template <typename MailboxProvider, typename Profiler>
inline auto makeChangeComposer(MailboxProvider m, Profiler profiler) {
    return ChangeComposer<MailboxProvider, Profiler>(m, profiler);
}

} // namespace access_impl
} // namespace doberman

#endif /* DOBERMAN_SRC_ACCESS_IMPL_CHANGE_COMPOSER_H_ */
