#pragma once

#include <macs_pg/changelog/change.h>

#include <src/access_impl/wrap_yield.h>
#include <src/access_impl/timer.h>
#include <src/access_impl/change_composer.h>
#include <src/access_impl/wait_list.h>
#include <src/logic/change.h>

#include <src/detail/dereference.h>
#include <src/profiling/profiler.h>
#include <src/meta/types.h>
#include <boost/range/adaptors.hpp>
#include <boost/variant.hpp>

namespace doberman {
namespace access_impl {

template <typename Shard, typename ChangeComposer, typename Profiler, typename Yield = boost::asio::yield_context>
class ChangeLogCache {
public:
    class Proxy {
    public:
        using Change = ::doberman::logic::Change;
        using ChangePtr = std::shared_ptr<const Change>;
        using Substitute = std::tuple<const ChangeComposer*, ::macs::Change>;
        using Value = boost::variant<Substitute, WaitList*, ChangePtr>;

        Proxy(const ChangeComposer& composer, ::macs::Change changeLogEntry)
        : v(Substitute(std::addressof(composer), std::move(changeLogEntry))) {
        }

        ChangePtr get(Yield yield) {
            return boost::apply_visitor(Getter(v, yield), v);
        }

    private:
        struct Getter {
            Value& v;
            Yield& yield;

            Getter(Value& v, Yield& yield) : v(v), yield(yield) {}

            ChangePtr operator () (Substitute s) const {
                WaitList wl;
                v = std::addressof(wl);
                try {
                    const auto& compose = *std::get<const ChangeComposer*>(s);
                    auto retval = compose(std::get<::macs::Change>(s), yield);
                    v = retval;
                    wl.notify_all();
                    return retval;
                } catch (...) {
                    v = s;
                    wl.notify_all();
                    throw;
                }
            }

            ChangePtr operator () (WaitList* wl) const {
                wl->wait(yield);
                return boost::apply_visitor(*this, v);
            }

            ChangePtr operator () (const ChangePtr& ptr) const {
                return ptr;
            }
        };

        Value v;
    };

    ChangeLogCache(Shard shard, ChangeComposer composer, Profiler profiler)
    : shard_(std::move(shard)), composer_(std::move(composer)),
      profiler_(std::move(profiler)) {}

    using Change = Proxy;
    using ChangePtr = std::shared_ptr<Change>;

    ChangePtr getChange(ChangeId id) const {
        const auto i = cache_.find(id);
        if (i == cache_.end()) {
            return nullptr;
        }
        return i->second;
    }

    template <typename SortedIdSequence>
    void update(const SortedIdSequence& ids, Yield yield) {
        cleanup(ids);
        const auto ids2fetch = absent(ids);
        if (!ids2fetch.empty()) {
            fetch(ids2fetch, yield);
        }
    }

private:
    using Cache = std::map<ChangeId, ChangePtr>;

    template <typename SortedIdSequence>
    auto absent(const SortedIdSequence& ids) {
        std::vector<ChangeId> retval;
        boost::set_difference(ids, cache_ | boost::adaptors::map_keys, std::back_inserter(retval));
        return retval;
    }

    void fetch(const std::vector<ChangeId>& ids, Yield yield) {
        const auto start = profiler().now();
        const auto changes = repo().getChanges(ids, wrap(yield));
        boost::transform(changes, std::inserter(cache_, cache_.end()),
                [&](auto& change) {
            const auto cid = change.changeId();
            return std::make_pair(cid, std::make_shared<Change>(composer_, std::move(change)));
        });
        profiler().write("change_log_cache", "fetch",
                std::to_string(changes.size()), profiler().passed(start));
    }

    template <typename SortedIdSequence>
    void cleanup(const SortedIdSequence& ids) {
        const auto used = [](const auto& v) {
            return v.second.use_count() > 1;
        };
        const auto requested = [&](const auto& v) {
            return boost::binary_search(ids, v.first);
        };
        const auto eraseIfUnused = [&](auto i) {
            return !(used(*i) || requested(*i)) ? cache_.erase(i) : std::next(i);
        };

        for (auto i = cache_.begin(); i != cache_.end(); i = eraseIfUnused(i));
    }

    auto& repo() const {
        return ::doberman::detail::dereference(shard_).changeLog();
    }

    auto& profiler() const {
        return ::doberman::detail::dereference(profiler_);
    }

    Shard shard_;
    ChangeComposer composer_;
    Profiler profiler_;
    Cache cache_;
};

} // namespace access_impl
} // namespace doberman
