#ifndef DOBERMAN_SRC_ACCESS_IMPL_CHANGE_QUEUE_H_
#define DOBERMAN_SRC_ACCESS_IMPL_CHANGE_QUEUE_H_

#include <queue>

#include <macs_pg/changelog/change.h>

#include <src/access_impl/wrap_yield.h>
#include <src/access_impl/timer.h>

#include <src/detail/dereference.h>
#include <src/meta/types.h>
#include <src/access_impl/change_composer.h>

#include <boost/fusion/adapted/struct/define_struct.hpp>


BOOST_FUSION_DEFINE_STRUCT((doberman)(access_impl), ChangeQueueTimes,
        (size_t, pull_tries)
        (std::uint64_t, sleep_seconds)
)


namespace doberman {
namespace access_impl {

template <typename ChangeCache>
class ChangeQueue {
    ChangeCache* cache_;
    const ChangeQueueTimes times_;
public:

    ChangeQueue(ChangeCache& cache, ChangeQueueTimes times)
    : cache_(&cache),
      times_(times)
        {}

    auto makeContext(const SubscriptionId& id) const {
        return id;
    }

    template <typename Ctx, typename Yield>
    auto getTopChange(const Ctx& ctx, Yield yield) const {
        return repo().get(uid(ctx), subscriptionId(ctx), yield);
    }

    template <typename Ctx, typename Yield>
    auto top(const Ctx& ctx, Yield yield) const {
        for (size_t tryNum = 1; tryNum < pullTries(); ++tryNum) {
            if (auto change = getTopChange(ctx, yield)) {
                return change;
            }
            timer::wait(sleepTime(), yield);
        }
        return getTopChange(ctx, yield);
    }

    template <typename Ctx, typename Yield>
    void pop(const Ctx& ctx, Yield yield) const {
        auto change = top(ctx, yield);
        if (!change) {
            throw std::logic_error(
                std::string(__PRETTY_FUNCTION__) + " Try pop from empty queue!");
        }
        repo().remove(uid(ctx), subscriptionId(ctx), change->id(), yield);
    }

private:
    auto& repo() const {
        return *cache_;
    }

    template <typename Ctx>
    static auto& uid(const Ctx& ctx) {
        return ctx.uid;
    }
    template <typename Ctx>
    static auto& subscriptionId(const Ctx& ctx) {
        return ctx.id;
    }
    Seconds sleepTime() const {
        return Seconds(times_.sleep_seconds);
    }
    size_t pullTries() const {
        return times_.pull_tries;
    }
};


template <typename ChangeCache>
inline auto makeChangeQueue(ChangeCache& cache, ChangeQueueTimes times) {
    return ChangeQueue<std::decay_t<ChangeCache>>(cache, std::move(times));
}

} // namespace access_impl
} // namespace doberman

#endif /* DOBERMAN_SRC_ACCESS_IMPL_CHANGE_QUEUE_H_ */
