#pragma once

#include <src/logic/error.h>
#include <apq/error.hpp>
#include <macs_pg/error.h>
#include <mail_errors/error_code.h>

namespace doberman {
namespace access_impl {
namespace errc {

/**
 * Error conditions for access_impl level
 */
enum errc_t {
    communication_error = 1, // bound for communication errors
    readonly_error, // bound for DB read only state errors
    endpoint_error, // bound for DB end point lookup errors
    query_canceled, // bound for query cancellation errors
    user_is_not_here, // bound for errors of no user in shard
    lock_not_available, // bound for DB lock acquisition error
    internal_error, // bound for DB internal error
    temporary_error, // bound for temporary errors which can disappear during retries.
    admin_shutdown, // db session interrupted by administrator
};

} // namespace errc
} // namespace access_impl
} // namespace doberman

namespace boost {
namespace system {

template <>
struct is_error_condition_enum<doberman::access_impl::errc::errc_t> : std::true_type {};

} // namespace system
} // namespace boost

namespace doberman {
namespace access_impl {
namespace errc {

class category : public boost::system::error_category {
public:
    const char* name() const BOOST_NOEXCEPT override {
        return "doberman::access_impl::errc::category";
    }

    std::string message(int ev) const override {
        switch (ev) {
        case communication_error:
            return "communication error";
        case readonly_error:
            return "database in read-only state";
        case endpoint_error:
            return "no database end point found";
        case query_canceled:
            return "query has been cancelled";
        case user_is_not_here:
            return "user is not in the shard";
        case lock_not_available:
            return "lock not available";
        case internal_error:
            return "internal database error";
        case temporary_error:
            return "temporary error";
        case admin_shutdown:
            return "session interrupted by administrator";
        default:
            return "unknown error condition";
        }
    }

    bool equivalent(int code,
                    const boost::system::error_condition &condition
                    ) const noexcept override{
        return boost::system::error_category::equivalent(code, condition);
    }

    bool equivalent(const boost::system::error_code &code,
                    int condition
                    ) const noexcept override {
        switch (condition) {
        case temporary_error:
            return code == communication_error || code == readonly_error ||
                    code == endpoint_error || code == query_canceled ||
                    code == lock_not_available || code == internal_error ||
                    code == admin_shutdown;

        case communication_error:
            return code == pgg::errc::communicationError || code == apq::error::sqlstate::too_many_connections;

        case readonly_error:
            return code.category() == macs::pg::error::getReadonlyCategory();

        case endpoint_error:
            return code == macs::pg::error::noEndpointForRole;

        case query_canceled:
            return code == apq::error::sqlstate::query_canceled;

        case user_is_not_here:
            return code == apq::error::sqlstate::custom_error_mdba1;

        case lock_not_available:
            return code == apq::error::sqlstate::lock_not_available;

        case internal_error:
            return code == apq::error::sqlstate::internal_error;

        case admin_shutdown:
            return code == apq::error::sqlstate::admin_shutdown;

        default:
            return false;
        }
    }
};

inline const boost::system::error_category& get_category() {
    static category instance;
    return instance;
}

inline boost::system::error_condition make_error_condition(errc_t e) {
    return boost::system::error_condition(static_cast<int>(e), get_category());
}

} // namespace errc
} // namespace access_impl
} // namespace doberman
