#ifndef DOBERMAN_SRC_ACCESS_IMPL_FACTORY_H_
#define DOBERMAN_SRC_ACCESS_IMPL_FACTORY_H_

#include <src/access_impl/change_queue.h>
#include <src/access_impl/shared_folder.h>
#include <src/access_impl/subscribed_folder.h>
#include <src/access_impl/subscription.h>
#include <src/access_impl/subscription_repository.h>
#include <src/detail/magic_hat.h>
#include <src/access_impl/change_cache.h>
#include <src/access_impl/subscription_cache.h>
#include <src/access_impl/envelope_copier.h>
#include <src/access_impl/revision_cache.h>

#include <boost/fusion/adapted/struct/define_struct.hpp>


BOOST_FUSION_DEFINE_STRUCT((doberman)(access_impl), SleepTimes,
        (doberman::access_impl::ChangeQueueTimes, change_queue)
        (doberman::access_impl::ChangeCacheTimes, change_cache)
        (doberman::access_impl::SubscriptionRepositoryTimes, subscription_repository)
)


namespace doberman {
namespace access_impl {

template <typename Profiler, typename Log, typename SubscriptionResource,
    typename ServiceFactory>
class AccessFactory {
public:
    auto changeQueue() const {
        return makeChangeQueue(*changeCache(), sleepTimes_.change_queue);
    }

    auto sharedFolder() const {
        return makeSharedFolder(getMailboxProvider(), profiler_, envelopeChunkSize_);
    }

    auto subscribedFolder() const {
        return makeSubscribedFolder(getMailboxProvider(), log_,
                retries_, workerId_, getRevisionCache(), profiler_,
                envelopeChunkSize_);
    }

    auto subscription() const {
        return makeSubscription(std::addressof(subscriptionCache()),
                std::addressof(getSubscriptionResource()));
    }

    auto subscriptionRepository() const {
        return makeSubscriptionRepository(std::addressof(subscriptionCache()),
                sleepTimes_.subscription_repository);
    }

    auto envelopeCopier() const {
        return RobustEnvelopeCopier<Log>(retries_, log_);
    }

    AccessFactory(
            ServiceFactory factory,
            Job& job,
            Profiler profiler,
            Log log,
            SleepTimes sleepTimes,
            Retries retries,
            SubscriptionResource subscriptionResource,
            std::size_t envelopeChunkSize)
    : factory_(factory),
      shardId_(job.shardId),
      workerId_(job),
      profiler_(profiler),
      log_(log),
      sleepTimes_(sleepTimes),
      retries_(retries),
      factoryCache_(std::make_shared<FactoryCache>(
              factory_, workerId_, shardId_,
              std::chrono::seconds(sleepTimes_.change_queue.sleep_seconds),
              std::chrono::seconds(sleepTimes_.change_cache.flush_removed_changes_seconds),
              std::chrono::seconds(sleepTimes_.subscription_repository.sleep_seconds),
              std::chrono::milliseconds(sleepTimes_.subscription_repository.least_wait_ms),
              std::move(subscriptionResource),
              profiler, log)),
       envelopeChunkSize_(envelopeChunkSize)
       {}

private:
    struct MailboxCreator {
        ServiceFactory factory_;
        auto operator()(Uid uid) const {
            return ::doberman::detail::dereference(factory_).mailbox(uid);
        }
    };

    auto getShardFactory() const {
        auto cache = factoryCache_;
        return [cache](auto&) {
            return cache->shard;
        };
    }

    auto& changeCache() const {
        return factoryCache_->changeCache;
    }

    auto& subscriptionCache() const {
        return factoryCache_->subscriptionCache;
    }

    auto getMailboxProvider() const {
        return factoryCache_->getMailboxProvider();
    }

    auto& getSubscriptionResource() const {
        return factoryCache_->subscriptionResource;
    }

    auto& getRevisionCache() const {
        return factoryCache_->revisionCache;
    }

    struct FactoryCache {
        using Mailboxes = ::doberman::detail::MagicHat<Uid, MailboxCreator>;
        struct MailboxProvider {
            Mailboxes& mailboxes;
            auto operator() (const Uid& uid) const {
                return mailboxes.get(uid);
            }
        };
        using ShardPtr = macs::ShardPtr;
        using ChangeComposer = ::doberman::access_impl::ChangeComposer<MailboxProvider, Profiler>;
        using ChangeLogCache = ::doberman::access_impl::ChangeLogCache<ShardPtr, ChangeComposer, Profiler>;
        using ChangeCache = ::doberman::access_impl::ChangeCache<ShardPtr, ChangeLogCache, Profiler, WorkerId, Log>;
        using SubscriptionCache = ::doberman::access_impl::SubscriptionCache<ShardPtr, WorkerId>;

        ShardPtr shard;
        Mailboxes mailboxes;
        std::shared_ptr<ChangeCache> changeCache;
        SubscriptionCache subscriptionCache;
        SubscriptionResource subscriptionResource;
        RevisionCache revisionCache;

        auto getMailboxProvider() { return MailboxProvider{mailboxes};}

        FactoryCache(ServiceFactory factory, WorkerId workerId, ShardId shardId,
                     typename ChangeCache::Duration changeTtl,
                     typename ChangeCache::Duration flushRemovedChangesTtl,
                     typename SubscriptionCache::Duration subscriptionTtl,
                     typename SubscriptionCache::Duration leastWait,
                     SubscriptionResource subscriptionResource,
                     Profiler profiler, Log log)
        : shard(::doberman::detail::dereference(factory).shard(shardId)),
          mailboxes({factory}),
          changeCache(std::make_shared<ChangeCache>(
                  workerId, shard, changeTtl, flushRemovedChangesTtl,
                  ChangeLogCache{shard, ChangeComposer{getMailboxProvider(), profiler}, profiler},
                  profiler, log)),
          subscriptionCache(workerId, shard, subscriptionTtl, leastWait),
          subscriptionResource(std::move(subscriptionResource)){}
    };

    ServiceFactory factory_;
    ShardId shardId_;
    WorkerId workerId_;
    Profiler profiler_;
    Log log_;
    const SleepTimes sleepTimes_;
    const Retries retries_;
    const std::shared_ptr<FactoryCache> factoryCache_;
    std::size_t envelopeChunkSize_;
};

template <typename Profiler, typename Log, typename ServiceFactory, typename SubscriptionResource>
inline auto makeAccessFactory(ServiceFactory factory, Job& job, Profiler profiler, Log log,
        SleepTimes sleepTimes, Retries retries, SubscriptionResource subscriptionResource, std::size_t envelopeChunkSize) {
    return AccessFactory<Profiler, Log, SubscriptionResource, ServiceFactory>(
            std::move(factory), job, std::move(profiler),
            std::move(log), std::move(sleepTimes), std::move(retries),
            std::move(subscriptionResource), envelopeChunkSize);
}

} // namespace access_impl
} // namespace doberman

#endif /* DOBERMAN_SRC_ACCESS_IMPL_FACTORY_H_ */
