#pragma once

#include <src/meta/types.h>
#include <src/logger/logger.h>
#include <src/detail/dereference.h>
#include <src/detail/finally.h>
#include <src/access_impl/timer.h>
#include <src/access_impl/wrap_yield.h>
#include <src/access_impl/error.h>

namespace doberman {
namespace access_impl {

template <
    typename Shard,
    typename RunStatus,
    typename WorkerIdControl,
    typename L,
    typename Duration>
inline auto makeHeartbeat(Shard shard, RunStatus& running,
        WorkerIdControl workerIdControl, L logger,
        Duration retryTimeout) {
    return [shard, &running, workerIdControl, logger, retryTimeout] (auto yield) mutable {
        auto& workerId = doberman::detail::dereference(workerIdControl);

        const auto f = detail::finally([&running, &workerId] {
            workerId.invalidate();
            running.reset();
        });
        const auto workerIdRecord = [&] {
            return std::make_tuple(
                log::worker_id=workerId.id(),
                log::worker_id_ttl=workerId.ttl(),
                log::worker_id_state=(workerId.valid() ? "valid" : "outdated"),
                log::launch_id=std::ref(workerId.launchId())
            );
        };
        try {
            while (running && workerId.valid()) {
                error_code ec;
                const bool valid = doberman::detail::dereference(shard)
                                    .subscriptions()
                                    .confirmTheJob(workerId.id(),
                                            to_string(workerId.launchId()),
                                            wrap(yield, ec));
                if (!ec && valid) {
                    if (workerId.validate()) {
                        DOBBY_LOG_NOTICE(logger, "worker_id has been validated successfull", workerIdRecord());
                        timer::wait(workerId.ttl()/2, yield);
                    } else {
                        DOBBY_LOG_ERROR(logger, "worker_id has been validated by database but timed out locally", workerIdRecord());
                        return;
                    }
                } else if (ec == errc::temporary_error) {
                    DOBBY_LOG_WARNING(logger, "worker_id has not been validated due to error and will be retried",
                            log::error_code=std::cref(ec), workerIdRecord());
                    timer::wait(retryTimeout, yield);
                } else {
                    if (ec) {
                        DOBBY_LOG_ERROR(logger, "worker_id has been invalidated due to non retriable error",
                                log::error_code=std::cref(ec), workerIdRecord());
                    } else {
                        DOBBY_LOG_ERROR(logger, "worker_id has not been validated by database", workerIdRecord());
                    }
                    return;
                }
            }
            DOBBY_LOG_WARNING(logger, "worker_id has not been validated due to timed out locally", workerIdRecord());
        } catch (const std::exception& e) {
            DOBBY_LOG_ERROR(logger, "worker_id can not been validated due to exception", workerIdRecord(), log::exception=e);
        }
    };
}
} // namespace access_impl
} // namespace doberman
