#pragma once

#include <src/access/job_finder.h>
#include <src/access_impl/wrap_yield.h>
#include <src/access_impl/timer.h>

#include <boost/range/adaptors.hpp>

namespace doberman {
namespace access_impl {

template <typename SharpeiClient, typename ShardProvider, typename Clock = std::chrono::steady_clock>
class JobFinder {
    using Duration = typename Clock::duration;
    using Timepoint = typename Clock::time_point;

    SharpeiClient sharpeiClient_;
    ShardProvider getShard_;
    Duration minDelay_;

    static auto now() { return Clock::now(); }
public:
    JobFinder(SharpeiClient sharpeiClient, ShardProvider getShard, Duration minDelay)
    : sharpeiClient_(std::move(sharpeiClient)),
      getShard_(std::move(getShard)),
      minDelay_(minDelay)
    {}

    auto makeContext() const {
        return std::make_tuple(now() - minDelay_);
    }

    template <typename Ctx, typename Yield>
    std::vector<ShardId> getShards(Ctx& ctx, Yield yield) const {
        holdMinDelay(ctx, yield);
        const auto stats = detail::dereference(sharpeiClient_).stat(wrap(yield));
        const auto retval = stats | boost::adaptors::map_keys;
        return {retval.begin(), retval.end()};
    }

    template <typename Ctx, typename Yield>
    boost::optional<Job> askForJob(Ctx&, const ShardId& id,
            const logic::FindJobParams& params, Yield yield) const {
        auto shard = getShard(id);
        if (auto workerId = detail::dereference(shard).subscriptions().findAJob(
                params.timeout, to_string(params.launchId), params.hostname,
                params.workerVersion, wrap(yield))) {
            return Job{
                    std::move(*workerId),
                    params.timeout,
                    std::chrono::steady_clock::now(),
                    params.launchId,
                    id};
        }
        return boost::none;
    }

private:
    auto getShard(const ShardId& id) const {
        return detail::dereference(getShard_)(id);
    }

    template <typename Ctx, typename Yield>
    void holdMinDelay(Ctx& ctx, Yield yield) const {
        auto& lastTime = std::get<0>(ctx);
        const auto delay = now() - lastTime;
        if (delay < minDelay_) {
            timer::wait(minDelay_ - delay, yield);
        }
        lastTime = now();
    }
};

template <typename SharpeiClient, typename ShardProvider, typename Clock = std::chrono::steady_clock>
inline auto makeJobFinder(SharpeiClient sharpeiClient, ShardProvider getShard,
        typename Clock::duration minDelay, const Clock& = std::chrono::steady_clock{}) {
    return JobFinder<SharpeiClient, ShardProvider, Clock>{
        std::move(sharpeiClient), std::move(getShard), minDelay};
}

} // nemaspece access_impl
} // namespace doberman
