#pragma once

#include <src/access_impl/timer.h>
#include <src/access_impl/wrap_yield.h>
#include <src/access_impl/error.h>
#include <src/meta/types.h>

#include <boost/fusion/adapted/struct/define_struct.hpp>

BOOST_FUSION_DEFINE_STRUCT((doberman)(access_impl), Retries,
        (size_t, retries)
        (std::uint64_t, timeout_sec)
)

namespace doberman {
namespace access_impl {

template <typename Rule, typename Log>
inline auto makeRetry(std::size_t maxRetries, Seconds interval, Rule rule, Log log) {
    return [=](auto func, auto yield) {
        for (std::size_t tryNum = 0; ; ++tryNum) {
            try {
                return func(wrap(yield));
            } catch (const boost::system::system_error& e) {
                if (tryNum == maxRetries || !rule(e.code())) {
                    throw;
                }
                log(std::addressof(e));
                timer::wait(interval, yield);
            }
        };
    };
}

template <typename Log>
inline auto makeTemporaryErrorRetry(std::size_t maxRetries, Seconds interval, Log&& log) {
    return makeRetry(maxRetries, interval,
            [](const error_code& e) { return e == errc::temporary_error;},
            std::forward<Log>(log));
}

} // namespace access_impl
} // namespace doberman
