#ifndef DOBERMAN_SRC_ACCESS_IMPL_SHARED_FOLDER_H_
#define DOBERMAN_SRC_ACCESS_IMPL_SHARED_FOLDER_H_

#include <src/access/shared_folder.h>
#include <src/access_impl/wrap_yield.h>
#include <boost/lexical_cast.hpp>
#include <src/meta/labels.h>

namespace doberman {
namespace access_impl {

class ChunkedIterator : public boost::iterator_facade<ChunkedIterator,
                            ::macs::EnvelopeWithMimeChunk::value_type,
                            boost::single_pass_traversal_tag> {
    using Chunk = ::macs::EnvelopeWithMimeChunk;
    using Iter = Chunk::iterator;
    using Getter = std::function<Chunk()>;
    using Ctx = std::tuple<Chunk, Iter, Getter>;

    std::shared_ptr<Ctx> ctx_;

    friend class boost::iterator_core_access;

    void increment() {
        if (ctx_) {

            auto& i = std::get<Iter>(*ctx_);
            auto& chunk = std::get<Chunk>(*ctx_);

            if (i != chunk.end()) {
                ++i;
            }

            if (i == chunk.end()) {
                chunk = std::get<Getter>(*ctx_)();
                if (chunk.empty()) {
                    ctx_ = nullptr;
                } else {
                    i = chunk.begin();
                }
            }
        }
    }

    bool equal(ChunkedIterator const& other) const {
        return other.ctx_ == ctx_;
    }

    auto& dereference() const { return *std::get<Iter>(*ctx_); }
public:
    ChunkedIterator() = default;
    template <typename ChunkGetter>
    ChunkedIterator(ChunkGetter next) : ctx_(std::make_shared<Ctx>()) {
        std::get<Iter>(*ctx_) = std::get<Chunk>(*ctx_).begin();
        std::get<Getter>(*ctx_) = std::move(next);
        increment();
    }
};

template <typename MailboxFactory, typename Profiler>
class SharedFolder {
    MailboxFactory factory_;
    Profiler profiler_;
    std::size_t chunkSize_;

    auto makeMailbox(const Uid& uid) const {
        return ::doberman::detail::dereference(factory_)(uid);
    }
public:
    using Coordinates = ::doberman::logic::SharedFolderCoordinates;

    SharedFolder(MailboxFactory factory,
                 Profiler profiler,
                 std::size_t chunkSize = 1000)
        : factory_(std::move(factory))
        , profiler_(std::move(profiler))
        , chunkSize_(chunkSize) {}

    auto makeContext(const Coordinates& c) const {
        return std::make_tuple(makeMailbox(c.owner.uid), c.fid);
    }

    template <typename Ctx, typename Yield>
    Revision revision(const Ctx& ctx, Yield&& yield) const {
        return folder(ctx, std::forward<Yield>(yield)).revision();
    }

    template <typename Ctx, typename Yield>
    auto envelopesWithMimes(const Ctx& ctx, int64_t lowestImapId, Yield&& yield) const {
        const auto f = folder(ctx, yield);

        auto next = [mailbox = mailbox(ctx), from = lowestImapId, to = f.imapUidNext(),
                     yield, fid = f.fid(), chunkSize = this->chunkSize_, profiler = this->profiler_] () mutable {

            const auto getEnvelopes = [&](auto yield) {
                return mailbox->envelopes().query().withMimes()
                        .byImapId(fid).from(from).to(static_cast<int64_t>(to)).count(chunkSize)
                        .get(yield);
            };

            const auto start = profiler.now();

            error_code ec;
            auto chunk = getEnvelopes(wrap(yield, ec));

            if (ec == macs::error::noSuchLabel) {
                mailbox->labels().resetLabelsCache();
                chunk = getEnvelopes(wrap(yield));
            } else if (ec) {
                throw system_error(ec);
            }

            profiler.write("query", "fetch_envelopes",
                           std::to_string(chunk.size()),
                           profiler.passed(start));

            if (!chunk.empty()) {
                auto& env = std::get<Envelope>(chunk.back());
                if (!boost::conversion::try_lexical_convert(env.imapId(), from)) {
                    throw std::invalid_argument(std::string("can not convert imapId ")
                            + env.imapId() + " to uint64_t");
                }
            }
            return chunk;
        };
        return boost::make_iterator_range(ChunkedIterator(next), ChunkedIterator());
    }

    template <typename Ctx, typename Yield>
    auto labels(const Ctx& ctx, Yield&& yield) const {
        const auto& repo = mailbox(ctx)->labels();
        const auto labels = repo.getAllLabels(wrap(yield));
        return meta::labels::LabelsCache(labels, [&repo, yield]{
            repo.resetLabelsCache();
            return repo.getAllLabels(wrap(yield));
        });
    }

private:
    template <typename Ctx, typename Yield>
    auto folder(const Ctx& ctx, Yield&& yield) const {
        const auto& repo = mailbox(ctx)->folders();
        repo.resetFoldersCache();
        return repo.getAllFolders(wrap(yield)).at(fid(ctx));
    }

    template <typename Ctx>
    static auto& mailbox(const Ctx& ctx) {
        return std::get<0>(ctx);
    }
    template <typename Ctx>
    static auto& fid(const Ctx& ctx) {
        return std::get<1>(ctx);
    }
};

template <typename MailboxFactory, typename Profiler>
inline auto makeSharedFolder(MailboxFactory f, Profiler profiler, std::size_t c) {
    return SharedFolder<MailboxFactory, Profiler>(std::move(f), std::move(profiler), c);
}

} // namespace access_impl
} // namespace doberman

#endif /* DOBERMAN_SRC_ACCESS_IMPL_SHARED_FOLDER_H_ */
