#ifndef DOBERMAN_SRC_ACCESS_IMPL_SUBSCRIBED_FOLDER_H
#define DOBERMAN_SRC_ACCESS_IMPL_SUBSCRIBED_FOLDER_H

#include <src/access/subscribed_folder.h>
#include <src/access_impl/wrap_yield.h>
#include <src/access_impl/retry.h>
#include <src/meta/types.h>
#include <src/logger/logger.h>


namespace doberman {
namespace access_impl {

template <typename MailboxFactory,
          typename Log,
          typename WorkerId,
          typename RevisionCache,
          typename Profiler>
class SubscribedFolder {
    MailboxFactory factory_;
    Log log_;
    Retries retries_;
    WorkerId workerId_;
    RevisionCache& cache_;
    Profiler profiler_;
    std::size_t chunkSize_;

    auto makeMailbox(const Uid& uid) const {
        return ::doberman::detail::dereference(factory_)(uid);
    }
public:
    using Coordinates = ::doberman::logic::SharedFolderCoordinates;
    using MsgCoordinates = ::doberman::logic::MessageCoordinates;

    SubscribedFolder(MailboxFactory factory, Log log,
            Retries retries, WorkerId workerId, RevisionCache& cache,
            Profiler profiler, std::size_t chunkSize = 1000)
    : factory_(std::move(factory)), log_(std::move(log)),
      retries_(std::move(retries)), workerId_(std::move(workerId)),
      cache_(cache), profiler_(std::move(profiler)), chunkSize_(chunkSize)
    {}

    auto makeContext(const Uid& uid) const {
        return std::make_tuple(makeMailbox(uid), uid);
    }

    template <typename Ctx, typename Yield>
    Revision revision(const Ctx& ctx, Coordinates coord, Yield&& yield) const {
        boost::optional<Revision> rev = cache_.get(uid(ctx), coord);
        if (!rev) {
            const auto logger = [coord, uid = uid(ctx), logger = log_] (auto ec) {
                DOBBY_LOG_NOTICE(logger, "retry getSyncedRevision()",
                        log::exception=*ec,
                        log::subscriber_uid=uid,
                        log::owner_uid=coord.owner.uid,
                        log::owner_fid=coord.fid);
            };
            const auto& repository = subscribedFolders(ctx);
            rev = retry(logger)([&](auto yield){
                return repository.getSyncedRevision(
                    coord.owner.uid, coord.fid, yield);
            }, std::forward<Yield>(yield));
            if (rev) {
                cache_.set(uid(ctx), coord, *rev);
            }
        }

        if (rev) {
            return *rev;
        } else {
            throw macs::system_error(error_code{macs::error::noSuchFolder});
        }
    }

    template <typename Ctx, typename Yield>
    void put(const Ctx& ctx, Coordinates coord, EnvelopeWithMimes env, Yield&& yield) const {
        throwIfOutdated(workerId_);

        const auto& repository = subscribedFolders(ctx);
        const auto logger = makeMidChangeLogger("Retry put", uid(ctx),
                coord.owner.uid, coord.fid, std::get<0>(env).mid());
        macs::SyncMessage sync = retry(logger)([&](auto yld) {
            throwIfOutdated(workerId_);
            return repository.syncMessage(
                    coord.owner.uid, std::get<Envelope>(env), std::get<MimeParts>(env),
                    std::vector<macs::Hash>(), macs::Hash(), yld);
        }, std::forward<Yield>(yield));
        cache_.set(uid(ctx), coord, sync.ownerRevision);
    }

    template <typename Ctx, typename Yield>
    void initPut(const Ctx& ctx, Coordinates coord, EnvelopeWithMimes env, Yield&& yield) const {
        throwIfOutdated(workerId_);

        const auto& repository = subscribedFolders(ctx);
        const auto logger = makeMidChangeLogger("Retry put", uid(ctx),
                coord.owner.uid, coord.fid, std::get<0>(env).mid());
        macs::SyncMessage sync = retry(logger)([&](auto yld) {
            throwIfOutdated(workerId_);
            const auto start = this->profiler_.now();
            auto res = repository.syncMessageQuiet(
                    coord.owner.uid, std::get<Envelope>(env), std::get<MimeParts>(env),
                    std::vector<macs::Hash>(), macs::Hash(), yld);
            this->profiler_.write("query", "init_store", this->profiler_.passed(start));
            return res;
        }, std::forward<Yield>(yield));
        cache_.set(uid(ctx), coord, sync.ownerRevision);
    }

    template <typename Ctx, typename Yield>
    void erase(const Ctx& ctx, Coordinates coord, ::macs::MidVec mids, Revision rev, Yield&& yield) const {
        throwIfOutdated(workerId_);

        const auto& repository = subscribedFolders(ctx);
        const auto logger = makeMidsChangeLogger("Retry erase", uid(ctx),
                coord.owner.uid, coord.fid, mids);
        macs::UpdateMessagesResult sync = retry(logger)([&](auto yld) {
            throwIfOutdated(workerId_);
            return repository.deleteMessages(
                    coord.owner.uid, coord.fid, mids, rev, yld);
        }, std::forward<Yield>(yield));
        cache_.set(uid(ctx), coord, sync.rev);
    }

    template <typename Ctx, typename Yield>
    void mark(const Ctx& ctx, MsgCoordinates msgCoord, std::vector<Label> labels, Revision rev, Yield&& yield) const {
        throwIfOutdated(workerId_);

        if (labels.empty()) {
            return;
        }

        const auto& repository = subscribedFolders(ctx);
        const auto logger = makeMidChangeLogger("Retry mark", uid(ctx),
                msgCoord.folder.owner.uid, msgCoord.folder.fid, msgCoord.mid);
        macs::UpdateMessagesResult sync = retry(logger)([&](auto yld) {
            throwIfOutdated(workerId_);
            return repository.labelMessages(
                    msgCoord.folder.owner.uid, msgCoord.folder.fid,
                    {msgCoord.mid}, rev, labels, yld);
        }, std::forward<Yield>(yield));
        cache_.set(uid(ctx), msgCoord.folder, sync.rev);
    }

    template <typename Ctx, typename Yield>
    void unmark(const Ctx& ctx, MsgCoordinates msgCoord, std::vector<Label> labels, Revision rev, Yield&& yield) const {
        throwIfOutdated(workerId_);

        if (labels.empty()) {
            return;
        }

        const auto& repository = subscribedFolders(ctx);
        const auto logger = makeMidChangeLogger("Retry unmark", uid(ctx),
                msgCoord.folder.owner.uid, msgCoord.folder.fid, msgCoord.mid);
        macs::UpdateMessagesResult sync = retry(logger)([&](auto yld) {
            throwIfOutdated(workerId_);
            return repository.unlabelMessages(
                    msgCoord.folder.owner.uid, msgCoord.folder.fid,
                    {msgCoord.mid}, rev, labels, yld);
        }, std::forward<Yield>(yield));
        cache_.set(uid(ctx), msgCoord.folder, sync.rev);
    }

    template <typename Ctx, typename Yield>
    void joinThreads(const Ctx& ctx, Coordinates coord,
                     ThreadId tid, std::vector<ThreadId> joinTids,
                     Revision rev, Yield&& yield) const {
        throwIfOutdated(workerId_);

        const auto& repository = subscribedFolders(ctx);
        const auto logger = makeTidChangeLogger("Retry join threads", uid(ctx),
                coord.owner.uid, coord.fid, tid);
        macs::UpdateMessagesResult sync = retry(logger)([&](auto yld) {
            throwIfOutdated(workerId_);
            return repository.joinThreads(coord.owner.uid, coord.fid,
                                   tid, joinTids, rev, yld);
        }, std::forward<Yield>(yield));
        cache_.set(uid(ctx), coord, sync.rev);
    }

    template <typename Ctx, typename Yield>
    std::vector<::macs::Envelope> envelopes(const Ctx& ctx, Coordinates coord, Yield&& yield) const {
        const auto logger = [coord, uid = uid(ctx), logger = log_] (auto ec) {
                DOBBY_LOG_NOTICE(logger, "retry getEnvelopes()",
                        log::exception=*ec,
                        log::subscriber_uid=uid,
                        log::owner_uid=coord.owner.uid,
                        log::owner_fid=coord.fid);
            };
        const auto& repository = subscribedFolders(ctx);
        return retry(logger)([&](auto yield) {
            return repository.getEnvelopes(coord.owner.uid, coord.fid, yield);
        }, std::forward<Yield>(yield));
    }

    template <typename Ctx, typename Yield>
    LabelSet labels(const Ctx& ctx, Yield&& yield) const {
        const auto& repo = mailbox(ctx)->labels();

        const auto logger = [uid = uid(ctx), logger = log_] (auto ec) {
            DOBBY_LOG_NOTICE(logger, "retry labels()",
                    log::exception=*ec, log::uid=uid);
        };

        return retry(logger)([&](auto yield){return repo.getAllLabels(yield);},
                std::forward<Yield>(yield));
    }

    template <typename Ctx, typename Yield>
    Label createLabel(const Ctx& ctx, Label label, Yield&& yield) const {
        throwIfOutdated(workerId_);

        const auto& repo = mailbox(ctx)->labels();

        const auto logger = [uid = uid(ctx), logger = log_] (auto ec) {
            DOBBY_LOG_NOTICE(logger, "retry getOrCreateLabel()",
                    log::exception=*ec, log::uid=uid);
        };

        return retry(logger)([&](auto yield){
            throwIfOutdated(workerId_);
            return repo.getOrCreateLabel(
                    label.name(), label.color(), label.type(), yield);},
            std::forward<Yield>(yield));
    }

    template <typename Ctx, typename Yield>
    void clear(const Ctx& ctx, Coordinates coord, Yield&& yield) const {
        throwIfOutdated(workerId_);

        const auto logger = [uid = uid(ctx), coord, logger = log_] (auto ec) {
            DOBBY_LOG_NOTICE(logger, "retry clearChunk()",
                    log::exception=*ec,
                    log::subscriber_uid=uid,
                    log::owner_uid=coord.owner.uid,
                    log::owner_fid=coord.fid);

        };

        std::size_t deletedMidsCount = 0;
        do {
            deletedMidsCount = retry(logger)([&, mbox = mailbox(ctx)](auto yld) {
                throwIfOutdated(workerId_);
                const auto start = this->profiler_.now();
                auto mids = mbox->subscribedFolders().getSyncedMids(
                            coord.owner.uid, coord.fid, this->chunkSize_, yld);
                if (!mids.empty()) {
                    mbox->envelopes().remove(mids, yld);
                }
                this->profiler_.write("query", "clear", std::to_string(mids.size()),
                                      this->profiler_.passed(start));
                return mids.size();
            }, std::forward<Yield>(yield));
        } while (deletedMidsCount != 0);
    }

    template <typename Ctx, typename Yield>
    int64_t lastSyncedImapId(const Ctx& ctx, Coordinates c, Yield&& yield) const {
        const auto logger = [uid = uid(ctx), logger = log_] (auto ec) {
            DOBBY_LOG_NOTICE(logger, "retry getLastSyncedImapId()",
                    log::exception=*ec, log::uid=uid);
        };

        return retry(logger)([&](auto yield){
            return mailbox(ctx)->subscribedFolders().getLastSyncedImapId(
                            c.owner.uid, c.fid, yield);
        }, std::forward<Yield>(yield));
    }

private:
    template <typename Ctx>
    static auto& uid(const Ctx& ctx) {
        return std::get<1>(ctx);
    }
    template <typename Ctx>
    static auto& mailbox(const Ctx& ctx) {
        return std::get<0>(ctx);
    }
    template <typename Ctx>
    static auto& subscribedFolders(const Ctx& ctx) {
        return mailbox(ctx)->subscribedFolders();
    }

    Seconds timeout() const {
        return Seconds(retries_.timeout_sec);
    }
    size_t retries() const {
        return retries_.retries;
    }

    template <typename Logger>
    auto retry(Logger logger) const {
        return makeTemporaryErrorRetry(retries(), timeout(), std::move(logger));
    }

    auto makeMidChangeLogger(const std::string& message,
                             const Uid& uid,
                             const Uid& ownerUid,
                             const Fid& ownerFid,
                             const Mid& mid) const {
        return [message, uid, ownerUid, ownerFid, mid, logger = log_] (auto ec) {
            DOBBY_LOG_NOTICE(logger, message,
                    log::exception=*ec,
                    log::subscriber_uid=uid,
                    log::owner_uid=ownerUid,
                    log::owner_fid=ownerFid,
                    log::mid=mid);
        };
    }

    auto makeMidsChangeLogger(const std::string& message,
                              const Uid& uid,
                              const Uid& ownerUid,
                              const Fid& ownerFid,
                              const ::macs::MidVec& mids) const {
        return [message, uid, ownerUid, ownerFid, mids, logger = log_] (auto ec) {
            DOBBY_LOG_NOTICE(logger, message,
                    log::exception=*ec,
                    log::subscriber_uid=uid,
                    log::owner_uid=ownerUid,
                    log::owner_fid=ownerFid,
                    log::mids=mids);
        };
    }

    auto makeTidChangeLogger(const std::string& message,
                             const Uid& uid,
                             const Uid& ownerUid,
                             const Fid& ownerFid,
                             const ThreadId& tid) const {
        return [message, uid, ownerUid, ownerFid, tid, logger = log_] (auto ec) {
            DOBBY_LOG_NOTICE(logger, message,
                    log::exception=*ec,
                    log::subscriber_uid=uid,
                    log::owner_uid=ownerUid,
                    log::owner_fid=ownerFid,
                    log::tid=tid);
        };
    }
};

template <typename MailboxFactory,
          typename Log,
          typename WorkerId,
          typename RevisionCache,
          typename Profiler>
inline auto makeSubscribedFolder(MailboxFactory factory, Log log,
        Retries retries, WorkerId workerId, RevisionCache& cache,
        Profiler profiler, std::size_t chunkSize) {
    return SubscribedFolder<MailboxFactory, Log, WorkerId, RevisionCache, Profiler>(std::move(factory),
            std::move(log), std::move(retries), std::move(workerId), cache, std::move(profiler), chunkSize);
}

} //namespace access_impl
} //namespace doberman

#endif // DOBERMAN_SRC_ACCESS_IMPL_SUBSCRIBED_FOLDER_H
