#ifndef DOBERMAN_SRC_ACCESS_IMPL_SUBSCRIPTION_H_
#define DOBERMAN_SRC_ACCESS_IMPL_SUBSCRIPTION_H_

#include <src/access/subscription.h>
#include <macs_pg/subscription/factory.h>
#include <macs_pg/subscription/subscription_action.h>
#include <src/access_impl/wrap_yield.h>
#include <src/access_impl/subscription_resource.h>

namespace doberman {
namespace access_impl {

template <typename IO, typename SubscriptionRes>
class Subscription {
    IO io_;
    SubscriptionRes subscriptionResource_;

    decltype(auto) io() const {
        return ::doberman::detail::dereference(io_);
    }

public:
    using State = ::doberman::SubscriptionState;

    Subscription(IO io, SubscriptionRes subscriptionResource)
    : io_(io), subscriptionResource_(subscriptionResource) {}

    auto makeContext(const SubscriptionId& id, const logic::Subscriber& s) const {
        return std::make_tuple(id.uid, id.id, s);
    }

    template <typename Ctx, typename Yield>
    State state(const Ctx& ctx, Yield&& yield) const {
        return io().getById(uid(ctx), id(ctx), yield).state();
    }

    template <typename Ctx, typename Yield>
    auto init(const Ctx& ctx, Yield yield) const {
        auto handle = ::doberman::detail::dereference(subscriptionResource_).get(subscriber(ctx), yield);
        return std::make_tuple(
                transit(ctx, Action::initialization, yield),
                std::move(handle));
    }

    template <typename Ctx, typename Yield>
    State sync(const Ctx& ctx, Yield&& yield) const {
        return transit(ctx, Action::synchronization, std::forward<Yield>(yield));
    }

    template <typename Ctx, typename Yield>
    State fail(const Ctx& ctx, std::string message, Yield&& yield) const {
        return io().markFailed(uid(ctx), id(ctx), std::move(message), yield).state();
    }

    template <typename Ctx, typename Yield>
    State finish(const Ctx& ctx, Yield&& yield) const {
        return transit(ctx, Action::termination, std::forward<Yield>(yield));
    }

    template <typename Ctx, typename Yield>
    auto clear(const Ctx& ctx, Yield&& yield) const {
        auto handle = ::doberman::detail::dereference(subscriptionResource_).get(subscriber(ctx), yield);
        return std::make_tuple(
                transit(ctx, Action::clearing, yield),
                std::move(handle));
    }

private:
    using Action = ::macs::pg::SubscriptionAction;

    template <typename Ctx>
    static auto& id(const Ctx& ctx) {
        return std::get<1>(ctx);
    }

    template <typename Ctx>
    static auto& uid(const Ctx& ctx) {
        return std::get<0>(ctx);
    }

    template <typename Ctx>
    static auto& subscriber(const Ctx& ctx) {
        return std::get<2>(ctx);
    }

    template <typename Ctx, typename Yield>
    State transit(const Ctx& ctx, Action action, Yield yield) const {
        return io().transitState(uid(ctx), id(ctx), action, yield).state();
    }
};

template <typename IO, typename SubscriptionRes>
inline auto makeSubscription(IO io, SubscriptionRes subscriptionResource) {
    return Subscription<IO, SubscriptionRes>(std::move(io), std::move(subscriptionResource));
}

} // namespace access_impl
} // namespace doberman

#endif /* DOBERMAN_SRC_ACCESS_IMPL_SUBSCRIPTION_H_ */
