#ifndef DOBERMAN_SRC_ACCESS_IMPL_SUBSCRIPTION_CACHE_H_
#define DOBERMAN_SRC_ACCESS_IMPL_SUBSCRIPTION_CACHE_H_

#include <macs_pg/subscription/subscription.h>
#include <macs_pg/subscription/subscription_action.h>

#include <src/access_impl/wrap_yield.h>
#include <src/access_impl/timer.h>
#include <src/access_impl/error.h>

#include <src/detail/dereference.h>
#include <src/meta/types.h>
#include <coroutine_mutex/coroutine_mutex.hpp>

#include <boost/range/adaptor/map.hpp>

namespace doberman {
namespace access_impl {

template <typename Shard,
          typename WorkerId,
          typename Mutex = ::coro::Mutex,
          typename Clock = std::chrono::steady_clock>
class SubscriptionCache {
public:
    using Yield = typename Mutex::YieldContext;
    using Duration = typename Clock::duration;

    SubscriptionCache(WorkerId workerId,
                      Shard shard,
                      Duration ttl,
                      Duration leastWait)
            : workerId_(std::move(workerId))
            , shard_(std::move(shard))
            , ttl_(ttl)
            , leastWait_(leastWait) {}

    auto getReserved(Yield yield) {
        throwIfOutdated(workerId_);

        ::coro::unique_lock<Mutex> l(fetchMutex_, yield, 0);
        fetchAll(yield);
        l.unlock();

        const auto v = cache_ | boost::adaptors::map_values;
        return std::vector<::macs::Subscription>{ v.begin(), v.end() };
    }

    auto reserve(std::size_t limit, Yield yield) {
        throwIfOutdated(workerId_);

        ::coro::unique_lock<Mutex> l(fetchMutex_, yield, 0);
        try {
            auto items = repo().getFreeForWorker(workerId_, limit, wrap(yield));
            l.unlock();
            updateCache(items);
            return items;
        } catch (const boost::system::system_error& e) {
            if (e.code() != errc::temporary_error) {
                throw;
            }
        }
        return decltype(repo().getFreeForWorker(workerId_, limit, wrap(yield))){};
    }

    auto getById(Uid uid, ::macs::SubscriptionId sid, Yield yield) {
        throwIfOutdated(workerId_);

        if (expired(now())) {
            cache_.clear();
        }

        auto i = cache_.find(std::tie(uid, sid));
        if (i == cache_.end()) {
            suspend(yield);
            ::coro::unique_lock<Mutex> l(fetchMutex_, yield, 0);
            if (!fresh(now())) {
                fetchAll(yield);
            }
            l.unlock();
            i = cache_.find(std::tie(uid, sid));
        }
        if (i == cache_.end()) {
            throw std::logic_error(
                    "Cannot find subscription with id=" + std::to_string(sid) +
                    " for uid=" + uid );
        }
        return i->second;
    }

    void release(Uid uid, macs::SubscriptionId sid, Yield yield) {
        if (workerId_.valid()) {
            repo().release(uid, sid, workerId_, wrap(yield));
        }
        cache_.erase(std::tie(uid, sid));
    }

    auto markFailed(Uid uid, ::macs::SubscriptionId sid, std::string message, Yield yield) {
        throwIfOutdated(workerId_);

        auto item = repo().markFailed(uid, sid, std::move(message), wrap(yield));
        cache_[std::tie(item.uid(), item.subscriptionId())] = item;
        return item;
    }

    auto transitState(Uid uid, ::macs::SubscriptionId sid,
                      ::macs::pg::SubscriptionAction action, Yield yield) {
        throwIfOutdated(workerId_);

        auto item = repo().transitState(uid, sid, action, wrap(yield));
        cache_[std::tie(item.uid(), item.subscriptionId())] = item;
        return item;
    }

private:
    using Timepoint = typename Clock::time_point;

    auto sinceUpdate(Timepoint now) const {
        return now - lastUpdateTime_;
    }

    bool expired(Timepoint now) const {
        return ttl_ <= sinceUpdate(now);
    }

    bool fresh(Timepoint now) const {
        return leastWait_ > sinceUpdate(now);
    }

    void suspend(Yield yield) {
        auto tpoint = now();
        if (fresh(tpoint)) {
            timer::wait(leastWait_ - sinceUpdate(tpoint), yield);
        }
    }

    static auto now() { return Clock::now(); }

    template <typename Range>
    void updateCache(Range&& r) {
        for (const auto& i : r) {
            cache_[std::tie(i.uid(), i.subscriptionId())] = i;
        }
    }

    void fetchAll(Yield yield) {
        cache_.clear();
        updateCache(repo().getByWorker(workerId_, wrap(yield)));
        lastUpdateTime_ = now();
    }

    auto& repo() const {
        return ::doberman::detail::dereference(shard_).subscriptions();
    }

    using Cache = std::map<std::tuple<Uid, ::macs::SubscriptionId>, ::macs::Subscription>;

    WorkerId workerId_;
    Shard shard_;
    Cache cache_;
    Mutex fetchMutex_;
    Duration ttl_;
    Duration leastWait_;
    Timepoint lastUpdateTime_ = now() - ttl_;
};

} // namespace access_impl
} // namespace doberman

#endif /* DOBERMAN_SRC_ACCESS_IMPL_SUBSCRIPTION_CACHE_H_ */
